package gov.va.vamf.service.shifttransition.userpreferences;

import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp400BadRequestException;
import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp500InternalServerErrorException;
import gov.va.vamf.service.shifttransition.infrastructure.mongo.MongoWrapper;

import org.bson.types.ObjectId;
import org.jongo.MongoCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Strings;
import com.mongodb.MongoException;
import com.mongodb.WriteResult;

/**
 * Interface for user preferences mongodb collection.
 */
public class UserPreferencesRepository {
	public static final String COLLECTION_NAME = "user_preferences";

	private static Logger logger = LoggerFactory.getLogger(UserPreferencesRepository.class);

	MongoWrapper mongo = new MongoWrapper();

//----------------------------------------------------------------------------------------------------------------------
//---------------------------------GET Methods--------------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

	/**
	 * Retrieve a {@link MyPreferences} for the given staffId/siteId.  If one is not found, return a {@link MyPreferences}
	 * populated with just staffId/siteId.
	 */
	public MyPreferences get(String staffId, String siteId) {
		logger.debug("Getting User Preferences for:  UserId {}, SiteId {}.", staffId, siteId);
		
		MyPreferences myPrefs;
		
		try {
			MongoCollection collection = mongo.getCollection(COLLECTION_NAME);
			myPrefs = mongo.findOne(collection, "{staffId: '" + staffId + "', siteId: '" + siteId + "'}", MyPreferences.class);
		} catch (MongoException e) {
			logger.error("Unable to find user preferences.", e);
			throw new WebApp500InternalServerErrorException("Unable to find user preferences.  Please try again later.");
		}

		if (myPrefs == null)
			myPrefs = new MyPreferences(staffId, siteId);

		return myPrefs;
	}
	
	/**
	 * Find a MyPreferences who matches this ObjectId
	 */
	public MyPreferences findMyPreferencesById(String strId) {
		ObjectId id = new ObjectId(strId);
		MyPreferences retvalue = mongo.getCollection(COLLECTION_NAME).findOne(id).as(MyPreferences.class);
		return retvalue;
	}

//----------------------------------------------------------------------------------------------------------------------
//---------------------------------Insert Methods-----------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

	/**
	 * Once we validate that the object is able to be saved and that it's valid, then we'll save it.
	 */
	public MyPreferences insert(String staffId, String siteId, MyPreferences userPreferences) {
		validateUserPreferences(staffId, siteId, userPreferences);
		validateInsert(userPreferences);
		
		try {
			MyPreferences found = this.get(staffId, siteId);
			if (Strings.isNullOrEmpty(found._id) == false)
				throw new WebApp400BadRequestException("Cannot insert a record when one already exists.");
			
			ObjectId newId = ObjectId.get();
			userPreferences._id = newId.toString();
			
			//Save the record and make sure no errors occurred
			WriteResult wr = mongo.getCollection(COLLECTION_NAME).insert(userPreferences);
			String error = wr.getError();
			if (Strings.isNullOrEmpty(error) == false)
				throw new WebApp500InternalServerErrorException("Mongo DB: Unable to save record: " + error);
			MyPreferences retvalue = findMyPreferencesById(userPreferences._id);
			
			if (retvalue == null)
				throw new WebApp500InternalServerErrorException("Unable to find record that was just saved with id=" + newId);
			return retvalue;
		}
		catch (MongoException ex) {
			throw new WebApp500InternalServerErrorException("Mongo DB: Unable to save: " + ex.getMessage());
		}
	}

//----------------------------------------------------------------------------------------------------------------------
//---------------------------------Update Methods-----------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

	public MyPreferences update(String staffId, String siteId, MyPreferences userPreferences) {
		validateUserPreferences(staffId, siteId, userPreferences);
    	validateUpdate(userPreferences);
    	
        logger.debug("Saving nurse preferences: ");

		try {
			MongoCollection collection = mongo.getCollection(COLLECTION_NAME);
			
			//Update the record, make sure 1 record was affected, and make sure no errors occurred
			WriteResult wr = collection.save(userPreferences);
			int numDocsAffected = wr.getN();
			if (numDocsAffected == 0)
				throw new WebApp500InternalServerErrorException("Mongo DB: Unable to update record with id=" + userPreferences._id);
			String error = wr.getError();
			if (Strings.isNullOrEmpty(error) == false)
				throw new WebApp500InternalServerErrorException("Mongo DB: Unable to update record (" + userPreferences._id + "): " + error);
			
			MyPreferences retvalue = findMyPreferencesById(userPreferences._id);
			if (retvalue == null)
				throw new WebApp500InternalServerErrorException("Unable to find record that was just updated with id=" + userPreferences._id);
			
			return retvalue;
		} catch (MongoException e) {
			throw new WebApp500InternalServerErrorException("Unable to save changes to your user preferences.  Please try again later.");
		}
	}
	
//----------------------------------------------------------------------------------------------------------------------
//---------------------------------Validation Methods-------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

	private void validateInsert(MyPreferences userPreferences) {
		if (Strings.isNullOrEmpty(userPreferences._id) == false)
			throw new WebApp400BadRequestException("User preferences._id cannot be populated for a POST.");
	}

	private void validateUpdate(MyPreferences userPreferences) {
		if (Strings.isNullOrEmpty(userPreferences._id) == true)
			throw new WebApp400BadRequestException("User preferences._id cannot be empty for a PUT.");
	}
	
	private void validateUserPreferences(String userId, String siteId, MyPreferences userPreferences) {
		if (userPreferences == null)
			throw new WebApp400BadRequestException("User preferences cannot be null.");
		if (Strings.isNullOrEmpty(userPreferences.siteId))
			throw new WebApp400BadRequestException("User preferences.siteId cannot be empty.");
		if (Strings.isNullOrEmpty(userPreferences.staffId))
			throw new WebApp400BadRequestException("User preferences.staffId cannot be empty.");
	}
}
