package gov.va.vamf.service.shifttransition.tasks.representations;

import gov.va.vamf.service.shifttransition.application.representations.Namespace;
import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp400BadRequestException;
import gov.va.vamf.service.shifttransition.tasks.domain.ScheduleType;
import gov.va.vamf.service.shifttransition.tasks.domain.rules.parsers.CustomParser;
import gov.va.vamf.service.shifttransition.tasks.domain.rules.parsers.FrequencyParser;

import java.util.Date;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;


/**
 * Represents the data that can be updated for a task.  Also this data is saved for a new task.  This information
 * will replace existing information so clients should include all data even if it is unchanged.
 *
 * @see gov.va.vamf.service.shifttransition.tasks.representations.NewScheduledTask
 */
@XmlRootElement(name = "update_task", namespace = Namespace.ns)
public class UpdatedScheduledTask {
    /**
     * Optional task comment.
     */
    @XmlElement()
    public String comment;

    /**
     * Required task start date. The date must be an UTC timestamp.
     */
    @XmlElement(required = true)
    public Date startDate;

    /**
     * Optional task end date. The date must be an UTC timestamp and must be after start date.
     */
    @XmlElement()
    public Date endDate;

    /**
     * Optional reoccurring schedule type.  Should be null if not a reoccurring task.
     * Valid non null values include: Frequency, Custom or Pattern.
     */
    @XmlElement()
    public String scheduleType;

    /**
     * Represents the schedule frequency based on the schedule type.  Formatting is case sensitive.<br/><br/>
     *
     * <table border=1>
     * <tr><td>
     * For <b>Frequency</b>, the pattern should be &quot;I0+;D0-31;H0-23;M0-59&quot;.<br>
     * 'I' is the number of intervals from 0 to {@link FrequencyParser#MAX_INTERVALS} (should be zero if there is an endDate).<br/>
     * 'D' is day frequency from 0 to {@link FrequencyParser#MAX_DAYS}.<br/>
     * 'H' is hour frequency from 0 to {@link FrequencyParser#MAX_HOURS}.<br/>
     * 'M' is minute frequency from 0 to {@link FrequencyParser#MAX_MINUTES}.<br/>
     * Each parameter in the pattern is required even if it has a value of zero.<br/>
     * <b>Example:</b> I5;D0;H12;M0
     * </td></tr>
     *
     * <tr><td>
     * For <b>Custom</b>, the pattern should be &quot;timezone;time;time;time...|day;day;day...|week&quot;.<br/>
     * 'time' is in military time without a colon (hhmm).<br/>
     * 'day' is one of the following 3 character codes: All, Sun, Mon, Tue, Wed, Thu, Fri, Sat<br/>
     * 'week' is one of the following: Weekly, EveryOther, EveryThird, EveryFourth<br/>
     * <b>Example:</b> America/Chicago;1000;1430;1815;2300|Sun;Tue;Thu|Weekly<br/><br/>
     *
     * Timezone is required to correctly calculate the next due date for the task on the server which may or may not be
     * in the same timezone as clients. The timezone should be from the list of timezone supported by Java's calendar
     * object.  Make sure to select the correct timezone based on daylight savings time.  For example CST does not
     * account for daylight savings time so clients should use the 'America/Chicago' timezone to include daylight
     * savings time.  Using a GMT offset will not work for time calculations because the offset does not include
     * daylight savings time.
     * </td></tr>
     * <table>
     */
    @XmlElement()
    public String scheduleFrequency;

    public void validateInput() {
        if (startDate == null)
        	throw new WebApp400BadRequestException("Unable to save task. Required field start date missing.");

        if (scheduleType != null && scheduleFrequency == null)
        	throw new WebApp400BadRequestException("Unable to save task. Schedule information was not provided.");

        if (!ScheduleType.isValid(scheduleType))
        	throw new WebApp400BadRequestException("Unable to save task. Schedule information is not valid.");

        if (endDate != null && endDate.before(startDate))
        	throw new WebApp400BadRequestException("Unable to save task. End date is before start date.");
    }
}
