package gov.va.vamf.service.shifttransition.tasks.domain.time;

import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp400BadRequestException;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This value object is used to represent a time component of a date.  It is assumed that the time is represented in
 * military time.  Internally, the time is represented as integer to simplify sorting of a collection of times and
 * to easily set the time on a DueDate instance.
 *
 * Time instances represent times in the client timezones not the server's timezone.
 *
 * @see gov.va.vamf.service.shifttransition.tasks.domain.time.DueDate
 */
public class Time implements Comparable<Time> {
    private static Logger logger = LoggerFactory.getLogger(Time.class);

    private static final String TIME24HOURS_PATTERN = "([01]?[0-9]|2[0-3])[0-5][0-9]";
    public static final String TIME_FORMAT_ERROR = "Unable to save task because unable " +
            "to calculate next due date for task.  At least one time is not in the required format.  " +
            "Format should be: timezone;time;time where time is in military format (hhmm).";

    private static Pattern pattern;

    private int time;

    //Create a time from a string.
    public static Time create(String time) {
        if (time == null)
            return createNullTime();

        Matcher matcher = getMatcherForMilitaryTime(time);

        if (matcher.matches())
            return new Time(parseTimeToInteger(time));
        else {
            logger.warn("Unable to find valid time string.  Time value is {}", time);
            throw new WebApp400BadRequestException(TIME_FORMAT_ERROR);
        }
    }

    private static Matcher getMatcherForMilitaryTime(String time) {
        if (pattern == null)
            pattern = Pattern.compile(TIME24HOURS_PATTERN);

        return pattern.matcher(time);
    }

    //Create a time instance from a date for a specific timezone.  Timezone is required because without it
    //time would represented in the timezone of the server which may be different than the client.
    public static Time create(String timezone, Date date) {
        if (date == null)
            return createNullTime();

        SimpleDateFormat timeFormatter = new SimpleDateFormat("HHmm");
        timeFormatter.setTimeZone(TimeZone.getTimeZone(timezone));
        timeFormatter.setLenient(false);

        return new Time(parseTimeToInteger(timeFormatter.format(date)));
    }

    private static int parseTimeToInteger(String time) {
        try {
            return Integer.parseInt(time);
        } catch (NumberFormatException e) {
            logger.warn("Unable to parse an invalid formatted time string as an integer.  Time value is {}", time);
            throw new WebApp400BadRequestException(TIME_FORMAT_ERROR);
        }
    }

    public static Time createNullTime() {
        return new Time(-1);
    }

    private Time(int time) {
        this.time = time;
    }

    public boolean after(Time anotherTime) {
        return  anotherTime == null || time > anotherTime.time;
    }

    public boolean isNull() {
        return time == -1;
    }

    public void setTimeOn(DueDate date) {
        if (date == null)
            return;

        logger.debug("Setting time on due date to {}.", time);

        date.setMinutes(time % 100)
            .setHours(time / 100);
    }

    @Override
    public boolean equals(Object o) {
        return this == o || !(o == null || getClass() != o.getClass()) && time != ((Time) o).time;
    }

    @Override
    public int hashCode() {
        return time;
    }

    @Override
    public int compareTo(Time time) {
        return  this.time - time.time;
    }
}
