package gov.va.vamf.service.shifttransition.tasks.domain.time;

import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp400BadRequestException;
import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp500InternalServerErrorException;

import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

import org.joda.time.DateTimeConstants;
import org.joda.time.Days;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.primitives.Ints;

/**
 * DueDate encapsulate a calendar instance.  It is used by domain objects to perform operations on a date such as
 * setting the time and comparing to other DueDate instances.  Dates for this domain are compared at a minimum level
 * of minutes so DueDate instances are created with zeroed out seconds and milliseconds.
 *
 * Important:  This is NOT a value object as the setTime operation will modify the state of the internal calendar
 * instance.
 */
public class DueDate {
    private static Logger logger = LoggerFactory.getLogger(DueDate.class);
    private static final int WEEK_7_DAYS = 7;

    Calendar calendar = Calendar.getInstance();

    public DueDate(Date date) {
        setDate(date);
        removeSeconds();
    }

    public DueDate(Date date, String timezone) {
        calendar.setTimeZone(TimeZone.getTimeZone(timezone));
        setDate(date);
        removeSeconds();
    }

    private void setDate(Date date) {
        if (date == null) {
            logger.warn("Null date cannot be used for a due date.");
            throw new WebApp400BadRequestException("Cannot complete operation. Date value cannot be null");
        }

        calendar.setTime(date);
    }

    //Minimum compare of dates is minutes
    private void removeSeconds() {
        calendar.set(Calendar.MILLISECOND, 0);
        calendar.set(Calendar.SECOND, 0);
    }

    public DueDate addDays(int days) {
        if (days != 0)
            calendar.add(Calendar.DATE, days);

        return this;
    }

    public DueDate addHours(int hours) {
        if (hours != 0)
            calendar.add(Calendar.HOUR_OF_DAY, hours);

        return this;
    }

    public DueDate addMinutes(int minutes) {
        if (minutes != 0)
            calendar.add(Calendar.MINUTE, minutes);

        return this;
    }

    public Date get() {
        return calendar.getTime();
    }

    public DueDate setHours(int hours) {
        calendar.set(Calendar.HOUR_OF_DAY, hours);
        return this;
    }

    public DueDate setMinutes(int minutes) {
        calendar.set(Calendar.MINUTE, minutes);
        return this;
    }

    public boolean after(DueDate date) {
        return calendar.after(date.calendar);
    }

    public boolean before(DueDate date) {
        return calendar.before(date.calendar);
    }

    public boolean between(DueDate startDate, DueDate endDate) {
        return calendar.after(startDate.calendar) && calendar.before(endDate.calendar);
    }

    @Override
    public boolean equals(Object o) {
        return this == o || !(o == null || getClass() != o.getClass()) &&
                calendar.getTimeInMillis() == ((DueDate) o).calendar.getTimeInMillis();
    }

    @Override
    public int hashCode() {
        return calendar.hashCode();
    }

    @Override
    public String toString() {
        return calendar.getTime().toString();
    }

    /**
     * Determines how many days to add to this DueDate based on when the startDate of the task was, what type of week
     * the interval is and which days are in that week.
     * @param startDate - When the task started.
     * @param week - What is the interval for week.
     * @param days - Which days of the week are valid days.
     */
	public void addWeekAndDay(DueDate startDate, CustomTaskWeek week, CustomTaskDays[] days) {
		switch (week) {
			case Weekly:
				weekly(days);
				break;
			case EveryOther:
				findNextValidDay(14, startDate, days);
				break;
			case EveryThird:
				findNextValidDay(21, startDate, days);
				break;
			case EveryFourth:
				findNextValidDay(28, startDate, days);
				break;
		}
		
	}

	/**
	 * For Weekly, we keep adding one day at a time until we find that we are on a valid day.
     * 
     * @param days - Which days of the week are valid days.
	 */
	private void weekly(CustomTaskDays[] days) {
		int[] validDays = getValidDaysOfWeek(days);

		//A week only has 7 days so we will find at least one valid day within that time.
		for (int i=0; i<WEEK_7_DAYS; i++) {
			addDays(1); //Keep adding days until we get a day of the week that is valid.
			
			int dayOfWeek = calendar.get(Calendar.DAY_OF_WEEK);
			boolean valid = Ints.contains(validDays, dayOfWeek);
			if (valid)
				return;
		}

		throw new WebApp500InternalServerErrorException("Could not find a valid Date within a Week");
	}

	/**
	 * Changes this DueDate to the next valid day - calculated by using the startDate, which days of the week are valid,
	 * and what the current DueDate (this) is.
	 * 
	 * @param daysBetweenPeriods - How many days are there between periods of valid weeks.
	 * @param startDate - When the task started.
	 * @param days - Which days of the week are valid days.
	 */
	private void findNextValidDay(int daysBetweenPeriods, DueDate startDate, CustomTaskDays[] days) {
		//Increment the current day by one - next we'll see if it's on a valid day.
		this.addDays(1);
		
		//Calculate the number of periods to have occurred since the startDate (the difference of days between the start
		//and current dueDate divided by the daysBetweenPeriods).
		LocalDate startDateNoTime = new LocalDate(startDate.calendar.get(Calendar.YEAR), startDate.calendar.get(Calendar.MONTH)+1, startDate.calendar.get(Calendar.DATE));
		LocalDate lastDueDateNoTime = new LocalDate(this.calendar.get(Calendar.YEAR), this.calendar.get(Calendar.MONTH)+1, this.calendar.get(Calendar.DATE));
		int daysBetween = Days.daysBetween(startDateNoTime, lastDueDateNoTime).getDays();
		int numPeriods = daysBetween/daysBetweenPeriods;

		//Also get what days of the week are valid days.
		int[] validJodaDays = getValidDaysOfWeekJoda(days);
		
		//Now that we know how many periods have elapsed since the startDate, we can figure out when the closest
		//week of valid days will be.
		LocalDate validDay = startDateNoTime.plusDays(daysBetweenPeriods*numPeriods);
		
		//One of two situations can happen here: either the days are in this period or they are in the next period.
		//Check all of the days in this period to see if this current dueDate is one of them.
		for (int i=0; i<WEEK_7_DAYS; i++) {
			//As we check each of the possible valid days in this period, make sure the current valid day isn't before the current dueDate.
			if (validDay.isBefore(lastDueDateNoTime)) {
				validDay = validDay.plusDays(1);
				continue;
			}
			
			//Get the day of the week and see if it's a valid day.  If it is, add that many days to the current dueDate
			//and exit this method.  If it isn't, check the next valid day.
			if (updateIfDayOfWeekValid(validDay, validJodaDays, lastDueDateNoTime))
				return;
			validDay = validDay.plusDays(1);
		}
		
		//Because we didn't find a valid day by now, it means the next valid day is in the next period.
		//Grab the beginning of the next period (and just to be safe, validate that it's a valid day).
		validDay = startDateNoTime.plusDays(daysBetweenPeriods*(numPeriods+1)); //Get the next period.
		if (updateIfDayOfWeekValid(validDay, validJodaDays, lastDueDateNoTime))
			return;
		
		throw new WebApp500InternalServerErrorException("No valid Date could be found within 2 periods");
	}
	
	/**
	 * If the validDay is in validJodaDays, then we figure out how many days are between the validDay and the
	 * current dueDate.  We then add this many days to the current dueDate so it is the same date. 
	 * 
	 * @param validDay - The day we want to see if it's in validJodaDays.
	 * @param validJodaDays - Which days of the week are valid days (as obtained from {@link #getValidDaysOfWeekJoda(CustomTaskDays[])}).
	 * @param lastDueDateNoTime - Used to calculate how many days to add to the current dueDate.
	 * @return True if we updated the current dueDate.
	 */
	private boolean updateIfDayOfWeekValid(LocalDate validDay, int[] validJodaDays, LocalDate lastDueDateNoTime) {
		int dayOfWeek = validDay.dayOfWeek().get();
		boolean valid = Ints.contains(validJodaDays, dayOfWeek);
		if (valid) {
			//Note that daysBetween could be zero if the next valid day is what current dueDate is currently set to.
			int daysBetween = Days.daysBetween(lastDueDateNoTime, validDay).getDays();
			this.addDays(daysBetween);
			return true;
		}
		return false;
	}
	
	/**
	 * Takes an array of days and retrieves the value Calendar uses for the day of the week.  The returned array can
	 * then be queried using Ints.contains.
	 */
	private int[] getValidDaysOfWeek(CustomTaskDays[] days) {
		if (days == null || days.length == 0)
			throw new WebApp400BadRequestException("Custom Task Days must have at least one entry.");
		if (days.length == 1 && CustomTaskDays.All == days[0]) {
			int[] retvalue = new int[] {Calendar.SUNDAY, Calendar.MONDAY, Calendar.TUESDAY, Calendar.WEDNESDAY, Calendar.THURSDAY, Calendar.FRIDAY, Calendar.SATURDAY};
			return retvalue;
		}
		
		int[] retvalue = new int[days.length];
		for (int i=0; i<days.length; i++) {
			switch (days[i]) {
				case Sun:
					retvalue[i] = Calendar.SUNDAY;
					break;
				case Mon:
					retvalue[i] = Calendar.MONDAY;
					break;
				case Tue:
					retvalue[i] = Calendar.TUESDAY;
					break;
				case Wed:
					retvalue[i] = Calendar.WEDNESDAY;
					break;
				case Thu:
					retvalue[i] = Calendar.THURSDAY;
					break;
				case Fri:
					retvalue[i] = Calendar.FRIDAY;
					break;
				case Sat:
					retvalue[i] = Calendar.SATURDAY;
					break;
				case All:
					throw new WebApp400BadRequestException("Custom Task Days cannot include All with additional days of the week.");
			}
		}
		
		if (retvalue.length > WEEK_7_DAYS)
			throw new WebApp400BadRequestException("Custom Task Days cannot have more than 7 entries in it - there are only 7 days in a week.");
		
		return retvalue;
	}
	
	/**
	 * Converts the days into an array of integers similar to {@link #getValidDaysOfWeek(CustomTaskDays[])} but in the Joda format.
	 * Joda uses a value of 1 for Monday instead of Sunday and considers Monday the start of the week whereas Java's
	 * Calendar uses Sunday as a 1 and the start of the week.  The returned array can then be queried using Ints.contains.
	 */
	private int[] getValidDaysOfWeekJoda(CustomTaskDays[] days) {
		if (days == null || days.length == 0)
			throw new WebApp400BadRequestException("Custom Task Days must have at least one entry in it.");
		if (days.length == 1 && CustomTaskDays.All == days[0]) {
			int[] retvalue = new int[] {DateTimeConstants.SUNDAY, DateTimeConstants.MONDAY, DateTimeConstants.TUESDAY, DateTimeConstants.WEDNESDAY, DateTimeConstants.THURSDAY, DateTimeConstants.FRIDAY, DateTimeConstants.SATURDAY};
			return retvalue;
		}
		
		int[] retvalue = new int[days.length];
		for (int i=0; i<days.length; i++) {
			switch (days[i]) {
				case Sun:
					retvalue[i] = DateTimeConstants.SUNDAY;
					break;
				case Mon:
					retvalue[i] = DateTimeConstants.MONDAY;
					break;
				case Tue:
					retvalue[i] = DateTimeConstants.TUESDAY;
					break;
				case Wed:
					retvalue[i] = DateTimeConstants.WEDNESDAY;
					break;
				case Thu:
					retvalue[i] = DateTimeConstants.THURSDAY;
					break;
				case Fri:
					retvalue[i] = DateTimeConstants.FRIDAY;
					break;
				case Sat:
					retvalue[i] = DateTimeConstants.SATURDAY;
					break;
				case All:
					throw new WebApp400BadRequestException("Custom Task Days cannot include All with any additional days of the week.");
			}
		}
		
		if (retvalue.length > WEEK_7_DAYS)
			throw new WebApp400BadRequestException("Custom Task Days cannot have more than 7 entries in it; there are only 7 days in a week.");
		
		return retvalue;
	}
}
