package gov.va.vamf.service.shifttransition.tasks.domain.rules.parsers;

import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp400BadRequestException;
import gov.va.vamf.service.shifttransition.tasks.domain.time.DueDate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Frequency schedule type parser used to validate and read the data encoded in the scheduleFrequency property.
 * See the UpdatedScheduledTask representation for details on the string format.  Documentation is not duplicated
 * here to prevent it from ever getting out of sync with the representation.  Documentation is representation is
 * required to support client development.
 *
 * Values from the parser are used by a next due date rule to calculate the next due date for a task.
 */
public class FrequencyParser {
    private static Logger logger = LoggerFactory.getLogger(FrequencyParser.class);

    public static final int MAX_DAYS = 90;         //requirement max
    public static final int MAX_HOURS = 120;       //requirement max
    public static final int MAX_MINUTES = 59;      //requirement max
    public static final int MAX_INTERVALS = 100;   //requirement max

    private int daysToAdd;
    private int hoursToAdd;
    private int minutesToAdd;
    private int intervalCount;

    public void parse(String scheduleFrequency, boolean validate) {
        logger.debug("Parsing schedule frequency.");

        if (validate && scheduleFrequency == null) {
            logger.error("Schedule frequency is null. Clients must provide a schedule frequency when the schedule type is frequency.");

            throw new WebApp400BadRequestException("Unable to save task because unable " +
                    "to calculate next due date for task.  Schedule frequency is missing.");
        }

        logger.debug("Parsing schedule frequency {}.", scheduleFrequency);

        String[] periods = scheduleFrequency.split(";");

        if (validate)
            verifyPeriods(periods);

        intervalCount = getInterval(periods[0]);
        daysToAdd = getInterval(periods[1]);
        hoursToAdd = getInterval(periods[2]);
        minutesToAdd = getInterval(periods[3]);

        if (validate) {
        	verifyDayHoursMinutesSet();
            verifyRanges();
        }
    }

	private void verifyPeriods(String[] periods) {
        if (periods.length != 4 || !periods[0].startsWith("I") || !periods[1].startsWith("D") ||
                !periods[2].startsWith("H") || !periods[3].startsWith("M")) {

            logger.error("Client set an invalid period in the schedule frequency.");
            throw new WebApp400BadRequestException("Unable to save task because unable " +
                    "to calculate next due date for task.  Invalid frequency format.  " +
                    "Format should be: D0-" + MAX_DAYS + ";H0-" + MAX_HOURS + ";M0-" + MAX_MINUTES + ".");
        }
    }

    private int getInterval(String period) {
        try {
            return Integer.parseInt(period.substring(1));
        } catch (NumberFormatException e) {
            logger.error("Client set a non-numeric period value in the schedule frequency.");
            throw new WebApp400BadRequestException("Unable to save task because unable " +
                    "to calculate next due date for task.  Period is not a number.");
        }
    }

    private void verifyDayHoursMinutesSet() {
    	if (daysToAdd == 0 && hoursToAdd == 0 && minutesToAdd == 0) {
            logger.error("Client set an invalid value in the schedule frequency.");
            throw new WebApp400BadRequestException("Unable to save task because unable " +
                    "to calculate next due date for task.  At least one of Days/Hours/Minutes is required.");
        }
	}

    private void verifyRanges() {
        if (intervalCount < 0 || intervalCount > MAX_INTERVALS) {
            logger.error("Client set an invalid internal count value in the schedule frequency.");
            throw new WebApp400BadRequestException("Unable to save task because unable " +
                    "to calculate next due date for task.  Interval outside of valid range: 0-" + MAX_INTERVALS + ".");
        }

        if (daysToAdd < 0 || daysToAdd > MAX_DAYS) {
            logger.error("Client set an invalid day value in the schedule frequency.");
            throw new WebApp400BadRequestException("Unable to save task because unable " +
                    "to calculate next due date for task.  Days to add outside of valid range: 0-" + MAX_DAYS + ".");
        }

        if (hoursToAdd < 0 || hoursToAdd > MAX_HOURS) {
            logger.error("Client set an invalid hour value in the schedule frequency.");
            throw new WebApp400BadRequestException("Unable to save task because unable " +
                    "to calculate next due date for task.  Hours to add outside of valid range: 0-" + MAX_HOURS + ".");
        }

        if (minutesToAdd < 0 || minutesToAdd > MAX_MINUTES) {
            logger.error("Client set an invalid minute value in the schedule frequency.");
            throw new WebApp400BadRequestException("Unable to save task because unable " +
                    "to calculate next due date for task.  Minutes to add outside of valid range: 0-" + MAX_MINUTES + ".");
        }
    }

    public int intervalCount() {
        return intervalCount;
    }

    public void addFrequency(DueDate date) {
        logger.debug("Adding frequency to due date {}; days: {}, hours: {}, minutes: {}.", date, daysToAdd, hoursToAdd, minutesToAdd);

        date.addDays(daysToAdd)
            .addHours(hoursToAdd)
            .addMinutes(minutesToAdd);
    }
}
