package gov.va.vamf.service.shifttransition.tasks.domain.rules.parsers;

import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp500InternalServerErrorException;
import gov.va.vamf.service.shifttransition.tasks.domain.time.CustomTaskDays;
import gov.va.vamf.service.shifttransition.tasks.domain.time.CustomTaskWeek;
import gov.va.vamf.service.shifttransition.tasks.domain.time.DueDate;
import gov.va.vamf.service.shifttransition.tasks.domain.time.Time;

import java.util.Arrays;
import java.util.TimeZone;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Custom schedule type parser used to validate and read the data encoded in the scheduleFrequency property.
 * See the UpdatedScheduledTask representation for details on the string format.  Documentation is not duplicated
 * here to prevent it from ever getting out of sync with the representation.  Documentation is representation is
 * required to support client development.
 *
 * Values from the parser are used by a next due date rule to calculate the next due date for a task.
 */
public class CustomParser {
    private static Logger logger = LoggerFactory.getLogger(CustomParser.class);

    private Time[] timesForToday;
    private String timezone;
    private CustomTaskDays[] days;
    private CustomTaskWeek week;

//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------Parse Method-----------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

    public void parse(String scheduleFrequency, boolean validate) {
        logger.debug("Parsing schedule frequency.");

        if (validate && scheduleFrequency == null) {
            logger.error("Schedule frequency is null. Clients must provide a schedule frequency when the schedule type is custom.");

            throw new WebApp500InternalServerErrorException("Unable to save task because unable " +
                    "to calculate next due date for task.  Schedule frequency is missing.");
        }

        logger.debug("Parsing schedule frequency {}.", scheduleFrequency);
        
        String[] timesDaysWeeks = scheduleFrequency.split("\\|");
        if (validate && timesDaysWeeks.length != 3) {
            logger.error("Schedule frequency did not contain all the required fields.");

            throw new WebApp500InternalServerErrorException("Unable to save task because unable " +
                    "to calculate next due date for task.  Schedule frequency is missing two pipe symbols.");
        }
        String[] times = timesDaysWeeks[0].split(";");
        String[] sDays = timesDaysWeeks[1].split(";");
        String sWeek = timesDaysWeeks[2];

        if (validate) {
            verifyTimes(times);
            verifyTimezone(times);
            verifyDays(sDays);
            verifyWeek(sWeek);
        }
        
        initDays(sDays);
        initWeek(sWeek);
        initTimesForToday(times);
    }
	
//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------Verify Methods---------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

	private void verifyTimes(String[] times) {
        if (times == null || times.length == 0) {
            logger.error("Client set no times or timezone in the schedule frequency.");
            throw new WebApp500InternalServerErrorException("Unable to save task because unable " +
                    "to calculate next due date for task.  Timezone information missing from frequency format.  " +
                    "Format should be: timezone;time;time where time is in military format (hhmm).");
        }

        if (times.length < 2) {
            logger.error("Client set no times in the schedule frequency.");
            throw new WebApp500InternalServerErrorException("Unable to save task because unable " +
                    "to calculate next due date for task.  At least one time is required but none found.  " +
                    "Format should be: timezone;time;time where time is in military format (hhmm).");
        }
    }

    private void verifyTimezone(String[] times) {
         for (String validTimezone : TimeZone.getAvailableIDs()) {
            if (validTimezone != null && validTimezone.equals(times[0])) {
                return;
            }
        }

        logger.error("Client set timezone to an unrecognized value in the schedule frequency.");

        throw new WebApp500InternalServerErrorException("Unable to save task because unable " +
                "to calculate next due date for task.  Timezone information is not valid.");
    }

	private void verifyWeek(String sWeek) {
		boolean valid = CustomTaskWeek.isValid(sWeek);
		if (valid == false) {
			logger.error("Client set no week in the schedule frequency.");
	        throw new WebApp500InternalServerErrorException("Unable to save task because unable " +
	                "to calculate next due date for task.  Week information missing from frequency format.");
		}
	}

	private void verifyDays(String[] sDays) {
		if (sDays == null || sDays.length == 0) {
			logger.error("Client set no day in the schedule frequency.");
	        throw new WebApp500InternalServerErrorException("Unable to save task because unable " +
	                "to calculate next due date for task.  Days information missing from frequency format.");
		}
		
		for (String day : sDays) {
			boolean valid = CustomTaskDays.isValid(day);
			if (valid == false) {
				logger.error("Client set Days to an unrecognized value in the schedule frequency.");
		        throw new WebApp500InternalServerErrorException("Unable to save task because unable " +
		                "to calculate next due date for task.  Days information is not valid.");
			}
        }
	}
	
//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------Init Methods-----------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

    private void initTimesForToday(String[] times) {
        logger.debug("Converting string times to int times and sorting.");

        timezone = times[0];

        timesForToday = new Time[times.length-1];

        for (int i = 1; i < times.length; i++) {
            timesForToday[i-1] = Time.create(times[i]);
        }
        Arrays.sort(timesForToday);
    }

    private void initWeek(String sWeek) {
		week = CustomTaskWeek.fromValue(sWeek);
	}

	private void initDays(String[] sDays) {
		days = new CustomTaskDays[sDays.length];
		for (int i=0; i<sDays.length; i++) {
			days[i] = CustomTaskDays.fromValue(sDays[i]);
		}
	}
	
//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------get Methods------------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

    public Time getTimeAfter(Time lastTime) {
        logger.debug("Getting next time.");

        //return this if lastTime was last time in array aka nothing after it.
        //don't want to return 1st time because it is important to know that the end of the day has been reached.
        Time nextTime = Time.createNullTime();

        for (int i = 0; i < timesForToday.length; i++) {
            if (timesForToday[i].after(lastTime)) {
                nextTime = timesForToday[i];
                break;
            }
        }

        return nextTime;
    }

    public Time getFirstTime() {
        return timesForToday[0];
    }
    
    public CustomTaskDays[] getDays() {
		return days;
	}

	public CustomTaskWeek getWeek() {
		return week;
	}

//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------create Methods---------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

    public Time createTimeInTimezone(DueDate date) {
        return Time.create(timezone, date.get());
    }

    public DueDate createDueDateInTimezone(DueDate date) {
        return new DueDate(date.get(), timezone);
    }
}
