package gov.va.vamf.service.shifttransition.tasks.domain;

import gov.va.vamf.service.shifttransition.tasks.domain.time.*;
import org.slf4j.*;

import java.util.Date;

/**
 * Available patient task statuses.
 */
public enum TaskStatus {
    //In urgency order. Do not change order unless urgency requirements change.
    Past_Due, Due, Pending, Completed, Dismissed, NA;

    private static Logger logger = LoggerFactory.getLogger(TaskStatus.class);

    //BR: Due is less than or equal to one half hour (30 minutes) before through the assigned time.
    //Pending is time greater than or equal to 31 minutes and up to 91 minutes BEFORE
    //the assigned time. Past Due is time greater than or equal to 1 minute AFTER the assigned time.
    public static TaskStatus status(DueDate nextDueDate) {
        Date now = new Date();

        TaskStatus status;

        if (nextDueDate == null)
            status = NA;
        else if (new DueDate(now).after(nextDueDate))
            status = Past_Due;
        else if (nextDueDate.before(new DueDate(now).addMinutes(31)))
            status = Due;
        else if (nextDueDate.between(new DueDate(now).addMinutes(30), new DueDate(now).addMinutes(91)))
            status = Pending;
        else
            status = NA;

        logger.debug("Task status is {} for nextDueDate of {} based on the current date/time of {}.", status, nextDueDate, now);

        return status;
    }

    public static boolean isValidCompleteStatus(String taskStatus) {
        return taskStatus.equalsIgnoreCase(Completed.name()) || taskStatus.equalsIgnoreCase(Dismissed.name());
    }

    //Returns true if either myNextDueDate has a more urgent status or it has the same status but the due date is sooner.
    public static boolean moreUrgent(DueDate myNextDueDate, DueDate otherNextDueDate) {
        TaskStatus thisStatus = status(myNextDueDate);
        TaskStatus otherStatus = status(otherNextDueDate);

        return thisStatus.moreUrgent(otherStatus) ||
                thisStatus == otherStatus && myNextDueDate.before(otherNextDueDate);
    }

    public boolean moreUrgent(TaskStatus other) {
        return ordinal() < other.ordinal();
    }

    @Override
    public String toString() {
        if (this == Past_Due)
            return "Past Due";
        else if (this == NA)
            return "N/A";
        else
            return name();
    }
}
