package gov.va.vamf.service.shifttransition.tasks.domain;

import gov.va.vamf.service.shifttransition.application.representations.Namespace;
import gov.va.vamf.service.shifttransition.tasks.domain.time.Range;
import gov.va.vamf.service.shifttransition.tasks.representations.NewScheduledTask;
import gov.va.vamf.service.shifttransition.tasks.representations.PatientTaskSummary;
import gov.va.vamf.service.shifttransition.tasks.representations.PatientTasksStatus;
import gov.va.vamf.service.shifttransition.tasks.representations.ScheduledTask;
import gov.va.vamf.service.shifttransition.tasks.representations.UpdatedScheduledTask;

import java.util.Date;
import java.util.List;
import java.util.UUID;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Entity the describes a patient task.  Once created only the comment and schedule may be changed.
 */
@XmlRootElement(name = "patient-task", namespace = Namespace.ns)
public class Task {
    private static Logger logger = LoggerFactory.getLogger(Task.class);

    @XmlElement()	
    private String id;                  //Unique id for task.

    @XmlElement()
    private String name;
    @XmlElement()
    private String type;                //Type of tasks like Custom or Clio.
    @XmlElement()
    private String externalTaskUrl;     //Url that is used by clients to get additional task display information.
    @XmlElement()
    private String comment;
    @XmlElement()
    private String createdBy;
    @XmlElement()
    private String lastModifiedBy;
    @XmlElement()
    private Date lastModifiedDate;
    @XmlElement()
    private Schedule schedule;

    public static Task createNullTask() {
        logger.debug("Creating null task.");

        NewScheduledTask newScheduleTask = new NewScheduledTask();
        newScheduleTask.name = "Null";
        newScheduleTask.type = "Null";
        newScheduleTask.startDate = new Date(Long.MAX_VALUE);

        return new Task(newScheduleTask, "");
    }

    public Task() {}

    public Task(NewScheduledTask newScheduledTask, String createdBy) {
        id = UUID.randomUUID().toString();

        logger.debug("Creating new task with id: {}.", id);

        name = newScheduledTask.name;
        type = newScheduledTask.type;
        externalTaskUrl = newScheduledTask.externalTaskUrl;
        this.createdBy = createdBy;
        lastModifiedDate = new Date();

        updateTask(newScheduledTask, null);
    }

    public boolean match(String taskId) {
        return id != null && taskId != null && id.equals(taskId);
    }

    public void updateTask(UpdatedScheduledTask task, String lastModifiedBy) {
        logger.debug("Updating task with id: {}.", id);

        comment = task.comment;

        //any schedule change cause a new Schedule which recalculates next due date based on start date.
        if (schedule == null || schedule.scheduleChanged(task))
            schedule = new Schedule(task);

        if (lastModifiedBy != null) {
            this.lastModifiedBy = lastModifiedBy;
            lastModifiedDate = new Date();
        }
    }

    public void addCurrentTaskSummary(Range range, List<PatientTaskSummary> patientTaskSummaries) {
        PatientTaskSummary summary = schedule.createPatientTaskSummary(range);

        summary.taskId = id;
        summary.taskName = name;

        patientTaskSummaries.add(summary);
    }

    public Task moreUrgent(Range range, Task other) {
        if (other == null)
            return this;

        logger.debug("Getting more urgent task for id: {}.", id);

        return schedule.moreUrgent(range, other.schedule) ? this : other;
    }

    public PatientTasksStatus generatePatientTasksStatus() {
        return schedule.generatePatientTasksStatus();
    }

    public ScheduledTask asScheduledTask() {
        ScheduledTask scheduledTask = new ScheduledTask();

        scheduledTask.taskId = id;
        scheduledTask.name = name;
        scheduledTask.type = type;
        scheduledTask.comment = comment;
        scheduledTask.externalTaskUrl = externalTaskUrl;
        scheduledTask.createdBy = createdBy;
        scheduledTask.lastModifiedBy = lastModifiedBy;
        scheduledTask.lastModifiedDate = lastModifiedDate;

        schedule.populateScheduleTask(scheduledTask);

        return scheduledTask;
    }

    public void setNextDueDate() {
        logger.debug("Setting next due date for task with id: {}.", id);
        schedule = schedule.getNextSchedule();
    }
}