package gov.va.vamf.service.shifttransition.tasks.domain;

import gov.va.vamf.service.shifttransition.application.representations.Namespace;
import gov.va.vamf.service.shifttransition.tasks.domain.rules.NextDueDateFactory;
import gov.va.vamf.service.shifttransition.tasks.domain.time.DueDate;
import gov.va.vamf.service.shifttransition.tasks.domain.time.Range;
import gov.va.vamf.service.shifttransition.tasks.representations.PatientTaskSummary;
import gov.va.vamf.service.shifttransition.tasks.representations.PatientTasksStatus;
import gov.va.vamf.service.shifttransition.tasks.representations.ScheduledTask;
import gov.va.vamf.service.shifttransition.tasks.representations.UpdatedScheduledTask;

import java.util.Date;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This value object represents the schedule information for a single task. See UpdatedScheduledTask representation
 * for a complete description of ScheduleFrequency.
 */
@XmlRootElement(name = "schedule", namespace = Namespace.ns)
public class Schedule {
    private static Logger logger = LoggerFactory.getLogger(Schedule.class);

    @XmlElement()
    private Date startDate;
    @XmlElement()
    private Date endDate;               //not required.  can be null.
    @XmlElement()
    private ScheduleType scheduleType;
    @XmlElement()
    private String scheduleFrequency;

    @XmlElement()
    private Date nextDueDate;           //calculated field. can be null.
    @XmlElement()
    private int numTimesScheduled;      //calculated field

    public Schedule() {}

    public Schedule(UpdatedScheduledTask task) {
        logger.debug("Creating new schedule.");

        startDate = task.startDate;
        endDate = task.endDate;
        scheduleType = ScheduleType.convertFromString(task.scheduleType);
        scheduleFrequency = task.scheduleFrequency;

        setNextDueDate();
    }

    public Schedule(Schedule schedule) {
        logger.debug("Replacing schedule.");

        startDate = schedule.startDate;
        endDate = schedule.endDate;
        nextDueDate = schedule.nextDueDate;
        scheduleType = schedule.scheduleType();
        scheduleFrequency = schedule.scheduleFrequency();
        numTimesScheduled = schedule.numTimesScheduled();

        setNextDueDate();
    }

    private void setNextDueDate() {
        logger.debug("Calculating next due date.");

        nextDueDate = new NextDueDateFactory().get(this).calculateNextDueDate();

        if (nextDueDate != null && endDate != null && nextDueDate().after(endDate()))
            nextDueDate = null;

        if (nextDueDate != null)
            numTimesScheduled ++;
    }

    public DueDate startDate() {
        return new DueDate(startDate);
    }

    public DueDate endDate() {
        return endDate == null ? null : new DueDate(endDate);
    }

    public DueDate nextDueDate() {
        return nextDueDate == null ? null : new DueDate(nextDueDate);
    }

    public int numTimesScheduled() {
        return numTimesScheduled;
    }

    public ScheduleType scheduleType() {
        return scheduleType;
    }

    public String scheduleFrequency() {
        return scheduleFrequency;
    }

    public Schedule getNextSchedule() {
        return new Schedule(this);
    }

    public PatientTaskSummary createPatientTaskSummary(Range range) {
        logger.debug("Creating patient task summary.");

        PatientTaskSummary summary = new PatientTaskSummary();

        if (!range.isInRange(nextDueDate()) || TaskStatus.status(nextDueDate()) == TaskStatus.NA) {
            summary.nextDueDate = null;
            summary.status = TaskStatus.NA.toString();
        } else {
            summary.nextDueDate = nextDueDate;
            summary.status = TaskStatus.status(nextDueDate()).toString();
        }

        return summary;
    }

    public boolean moreUrgent(Range range, Schedule other) {
        logger.debug("Determining more urgent schedule.");

        DueDate next = nextDueDate();

        return range.isInRange(next) && TaskStatus.moreUrgent(next, other.nextDueDate());
    }

    public PatientTasksStatus generatePatientTasksStatus() {
        logger.debug("Creating patient task status.");

        TaskStatus status = TaskStatus.status(nextDueDate());

        PatientTasksStatus patientTasksStatus = new PatientTasksStatus();
        patientTasksStatus.status = status.toString();

        if (status != TaskStatus.NA)
            patientTasksStatus.nextDueDate = nextDueDate;

        return patientTasksStatus;
    }

    public void populateScheduleTask(ScheduledTask scheduledTask) {
        scheduledTask.startDate = startDate;
        scheduledTask.endDate = endDate;
        scheduledTask.scheduleType = scheduleType.name();
        scheduledTask.scheduleFrequency = scheduleFrequency;
    }

    public boolean scheduleChanged(UpdatedScheduledTask task) {
        boolean changed = !startDate.equals(task.startDate) ||
                scheduleType != ScheduleType.convertFromString(task.scheduleType) ||
                !(scheduleFrequency == null ? task.scheduleFrequency == null : scheduleFrequency.equals(task.scheduleFrequency)) ||
                !(endDate == null ? task.endDate == null : endDate.equals(task.endDate));

        logger.debug("Schedule changed: {}", changed);

        return changed;
    }
}
