package gov.va.vamf.service.shifttransition.tasks.domain;

import gov.va.vamf.service.shifttransition.application.representations.Namespace;
import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp404NotFoundException;
import gov.va.vamf.service.shifttransition.tasks.domain.time.Range;
import gov.va.vamf.service.shifttransition.tasks.representations.NewScheduledTask;
import gov.va.vamf.service.shifttransition.tasks.representations.PatientTaskSummary;
import gov.va.vamf.service.shifttransition.tasks.representations.PatientTasksStatus;
import gov.va.vamf.service.shifttransition.tasks.representations.ScheduledTask;
import gov.va.vamf.service.shifttransition.tasks.representations.UpdatedScheduledTask;

import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.jongo.marshall.jackson.oid.ObjectId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

/**
 * Entity that holds all tasks defined and assigned to a patient.
 *
 * This does not include completed tasks which acts like documentation of a nurse's action.
 */
@XmlRootElement(name = "patient-tasks", namespace = Namespace.ns)
public class PatientTasks {
    private static Logger logger = LoggerFactory.getLogger(PatientTasks.class);

    //Mongodb unique key
    @ObjectId
    @XmlElement()
    private String _id;

    //patientId and siteId uniquely identify a patient.
    @XmlElement()
    private String patientId;

    
    //WARNING: The compiler cannot follow reflection methods which is what jongo uses to populate these patient tasks.
    //These variables are used and the app will breaks without it - there are queries that require those values
    @XmlElement()
	private String siteId;

    @XmlElement()
    private List<Task> tasks = new ArrayList<Task>();

    //Only one set of patient tasks should be active any single patient at a site (facility).
    //PatientTasks are deactivated by nightly process if they meet curtain criteria.
	@SuppressWarnings("unused")
	//WARNING: The compiler cannot follow reflection methods which is what jongo uses to populate these patient tasks.
	//These variables are used and the app will breaks without it - there are queries that require those values
	
	@XmlElement()
	private boolean activeVisit;

    public PatientTasks() {}

    public PatientTasks(String patientId, String siteId) {
        this.patientId = patientId;
        this.siteId = siteId;
        activeVisit = true;
    }

    public void deactivate() {
        activeVisit = false;
    }

    public Task addNewTask(NewScheduledTask newTask, String createdBy) {
        Task task = new Task(newTask, createdBy);
        tasks.add(task);

        return task;
    }

    public Task updateTask(UpdatedScheduledTask updatedTask, String taskId, String lastModifiedBy) {
        Task taskToUpdate = find(taskId);
        taskToUpdate.updateTask(updatedTask, lastModifiedBy);

        return taskToUpdate;
    }

    public ScheduledTask getTask(String taskId) {
        Task task = find(taskId);
        return task.asScheduledTask();
    }

    private Task find(String taskId) {
        logger.debug("Finding task by id: {}.", taskId);

        for (Task task : tasks)
            if (task.match(taskId))
                return task;

        logger.debug("Task with id {} not found.", taskId);
        throw new WebApp404NotFoundException("Task with id " + taskId + " not found.");
    }

    public void setNextDueDate(String taskId) {
        logger.debug("Setting next due date for task with id {}.", taskId);

        Task task = find(taskId);
        task.setNextDueDate();
    }

    public List<PatientTaskSummary> getAllTasks(Range range) {
        logger.debug("Getting all tasks as PatientTaskSummary instances.");

        List<PatientTaskSummary> summaries = Lists.newArrayList();

        for (Task task : tasks)
            task.addCurrentTaskSummary(range, summaries);

        addPatientId(summaries);

        return summaries;
    }

    private void addPatientId(List<PatientTaskSummary> summaries) {
        for (PatientTaskSummary summary : summaries)
            summary.patientId = patientId;
    }

    //get most urgent status from all tasks
    public PatientTasksStatus getMostUrgentTask(Range range) {
        logger.debug("Getting patient task status.");

        Task mostUrgentTask = findMostUrgentTask(range);

        PatientTasksStatus patientTasksStatus = mostUrgentTask.generatePatientTasksStatus();
        patientTasksStatus.patientId = patientId;

        return patientTasksStatus;
    }

    private Task findMostUrgentTask(Range range) {
        Task mostUrgentTask = Task.createNullTask();

        if (tasks.size() == 0)
            return mostUrgentTask;

        for (Task task : tasks)
            mostUrgentTask = task.moreUrgent(range, mostUrgentTask);

        return mostUrgentTask;
    }

    @Override
    public boolean equals(Object o) {
        return this == o || !(o == null || getClass() != o.getClass()) && _id.equals(((PatientTasks) o)._id);
    }

    @Override
    public int hashCode() {
        return _id.hashCode();
    }
}
