package gov.va.vamf.service.shifttransition.tasks;

import com.google.common.collect.Lists;

import gov.va.vamf.service.shifttransition.completedtasks.CompletedTask;
import gov.va.vamf.service.shifttransition.tasks.domain.*;
import gov.va.vamf.service.shifttransition.tasks.domain.time.Range;
import gov.va.vamf.service.shifttransition.tasks.representations.*;

import java.util.*;

/**
 * Domain service that provide domain functionality that does not fit well into other domain classes (entities, value objects).
 * This domain service is used for patient tasks.
 */
public class TaskTransformService {
    //convert completed tasks to patient task summary.  Some views return completed tasks with
    //patient tasks.
    public List<PatientTaskSummary> createFromCompletedTasks(List<CompletedTask> completedTasks) {
        List<PatientTaskSummary> patientTaskSummaries = Lists.newArrayList();

        if (completedTasks == null)
            return patientTaskSummaries;

        for (CompletedTask completedTask : completedTasks) {
            PatientTaskSummary summary = new PatientTaskSummary();
            summary.patientId = completedTask.patientId;
            summary.taskId = completedTask.taskId;
            summary.taskName = completedTask.taskName;
            summary.status = completedTask.status;
            summary.nextDueDate = completedTask.dateCompleted;

            patientTaskSummaries.add(summary);
        }

        return patientTaskSummaries;
    }

    //Add the most urgent task status for each patient.  Only one status per patient.
    public List<PatientTasksStatus> getMostUrgentTaskStatus(List<PatientTasks> multiplePatientTasks, Range range, List<String> patientIds) {
        List<PatientTasksStatus> patientTasksStatuses = Lists.newArrayList();
        List<String> usedPatientIds = Lists.newArrayList();

        for (PatientTasks patientTasks : multiplePatientTasks) {
            PatientTasksStatus tasksStatus = patientTasks.getMostUrgentTask(range);
            patientTasksStatuses.add(tasksStatus);
            usedPatientIds.add(tasksStatus.patientId);
        }

        addPatientsTasksStatusForPatientsWithoutTasks(patientIds, usedPatientIds, patientTasksStatuses);

        return patientTasksStatuses;
    }

    private void addPatientsTasksStatusForPatientsWithoutTasks(List<String> patientIds, List<String> usedPatientIds, List<PatientTasksStatus> patientTasksStatuses) {
        List<String> missingIds = Lists.newArrayList(patientIds);
        missingIds.removeAll(usedPatientIds);

        for (String missingId : missingIds) {
            patientTasksStatuses.add(getNAPatientTasksStatusForPatientId(missingId));
        }
    }

    private PatientTasksStatus getNAPatientTasksStatusForPatientId(String missingId) {
        PatientTasksStatus tasksStatus = new PatientTasksStatus();

        tasksStatus.patientId = missingId;
        tasksStatus.status = TaskStatus.NA.toString();

        return tasksStatus;
    }

    public List<PatientTaskSummary> getTasksSummariesWithoutNAStatusTask(List<PatientTasks> multiplePatientTasks, Range range) {
        List<PatientTaskSummary> patientTaskSummaries = getTasksSummaries(multiplePatientTasks, range);
        removeSummaries(patientTaskSummaries);

        return patientTaskSummaries;
    }

    //Get all patient tasks (not completed) for all patients.
    private List<PatientTaskSummary> getTasksSummaries(List<PatientTasks> multiplePatientTasks, Range range) {
        List<PatientTaskSummary> patientTaskSummaries = Lists.newArrayList();

        if (multiplePatientTasks == null)
            return patientTaskSummaries;

        for (PatientTasks patientTasks : multiplePatientTasks) {
            List<PatientTaskSummary> onePatientTaskSummaries = patientTasks.getAllTasks(range);

            patientTaskSummaries.addAll(onePatientTaskSummaries);
        }

        return patientTaskSummaries;
    }

    private void removeSummaries(List<PatientTaskSummary> patientTaskSummaries) {
        for (Iterator<PatientTaskSummary> iterator = patientTaskSummaries.iterator(); iterator.hasNext();) {
            PatientTaskSummary summary = iterator.next();

            if (summary.status.equals(TaskStatus.NA.toString()))
                iterator.remove();
        }
    }
}
