package gov.va.vamf.service.shifttransition.tasks;

import gov.va.vamf.service.shifttransition.application.resources.Routes;
import gov.va.vamf.service.shifttransition.application.repositories.RepositoryFactory;
import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp400BadRequestException;
import gov.va.vamf.service.shifttransition.tasks.representations.*;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Service;

import javax.ws.rs.*;
import javax.ws.rs.core.*;

import java.util.*;

/**
 * Resource used to save and retrieve tasks specific to a single patient.
 */
@Service
@Scope("request")
@Path("/nsc-service/facility/{DFN-site-id}/patients/{patient-id}/tasks")
public class PatientTasksResource {
    /**
     * Patient id that combine with a siteId (DFN-site-id) uniquely identifies a patient.
     */
    @PathParam("patient-id")
    private String patientId;

    /**
     * Site id that combine with a patient id uniquely identifies a patient.
     */
    @PathParam("DFN-site-id")
    private String siteId;

    private final RepositoryFactory repositoryFactory = new RepositoryFactory();

    /**
     * Returns a list of task summaries with the latest status for each task for a single patient within a given time frame.
     *
     * If a task has current status of NA it will be replaced with the latest completed or dismissed task info within the
     * time frame.  If there are no completed or dismissed tasked in the time frame then the task summary will not be
     * returned i.e. no tasks with a status of NA are returned.
     *
     * @param startDate Required query parameter.  The date must be an UTC timestamp.
     * @param endDate   Required query parameter.  The date must be an UTC timestamp.  This date must be after the start date.
     *
     * @return  PatientTaskSummaries which includes a list of PatientTaskSummary, one per task.
     */
    @GET
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public PatientTaskSummaries getPatientTasks(@Context UriInfo info, @Context HttpHeaders headers,
                                                @QueryParam("start-date") Long startDate, @QueryParam("end-date") Long endDate) {
    	if (null == startDate)
    		throw new WebApp400BadRequestException("start-date cannot be null");
    	if (null == endDate)
    		throw new WebApp400BadRequestException("end-date cannot be null");
    	
        PatientTaskService service = new PatientTaskService(repositoryFactory);

        PatientTaskSummaries patientTaskSummaries = service.getLatestTaskSummary(patientId, siteId, new Date(startDate), new Date(endDate));
        patientTaskSummaries.generateLinks(new Routes(info, siteId, patientId));

        return patientTaskSummaries;
    }

    /**
     * Saves a new task a single patient's tasks list.
     *
     * @param task  NewScheduledTask used to add a new task.
     *
     * @return  Returns task if save successful.  Returns a 400 error if any required data is missing or
     *          any of the information in the NewScheduledTask is not valid.
     */
    @POST
    @Consumes(MediaType.APPLICATION_JSON)
    public ScheduledTask saveNewTask(@Context UriInfo info, @Context HttpHeaders headers, NewScheduledTask task) {
        PatientTaskService service = new PatientTaskService(repositoryFactory);
        return service.saveNew(patientId, siteId, task);
    }

    /**
     * Returns all details for a single task.
     *
     * @param taskId    Id of task.
     * @return  ScheduleTask that represents a single task.  Returns a 404 error if the task is not found.
     */
    @GET
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    @Path("/{task-id}")
    public ScheduledTask getTask(@Context UriInfo info, @Context HttpHeaders headers, @PathParam("task-id") String taskId) {
        PatientTaskService service = new PatientTaskService(repositoryFactory);

        ScheduledTask task = service.getTask(taskId);
        task.generateLinks(new Routes(info, siteId, patientId));

        return task;
    }

    /**
     * Saves an update to a patient's exiting task.
     *
     * @param taskId    Id of task to update.
     * @param task      UpdatedScheduledTask that is used to update an existing task.
     *
     * @return  Task if update was successful or 404 error if the task is not found.
     */
    @PUT
    @Consumes(MediaType.APPLICATION_JSON)
    @Path("/{task-id}")
    public ScheduledTask updateTask(@Context UriInfo info, @Context HttpHeaders headers,
                               @PathParam("task-id") String taskId, UpdatedScheduledTask task) {
        PatientTaskService service = new PatientTaskService(repositoryFactory);
        return service.saveUpdate(patientId, siteId, taskId, task);
    }

}
