package gov.va.vamf.service.shifttransition.tasks;

import gov.va.vamf.service.shifttransition.application.repositories.AggregateResult;
import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp404NotFoundException;
import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp500InternalServerErrorException;
import gov.va.vamf.service.shifttransition.infrastructure.mongo.MongoWrapper;
import gov.va.vamf.service.shifttransition.tasks.domain.PatientTasks;

import java.util.Collections;
import java.util.List;

import org.jongo.MongoCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;
import com.mongodb.WriteConcern;

/**
 * Interface for patient tasks mongodb collection.
 */
public class PatientTasksRepository {
    public static final String COLLECTION_NAME = "patient_tasks";

    private static Logger logger = LoggerFactory.getLogger(PatientTasksRepository.class);

    MongoWrapper mongo = new MongoWrapper();

    public List<AggregateResult> getAllActivePatientTasksForSite(String siteId) {
        return  mongo.getCollection(COLLECTION_NAME)
                .aggregate("{$match: {siteId: #, activeVisit: true}}", siteId)
                .and("{$group: {_id: '$siteId', patientIds: {$push: '$patientId'}}}")
                .as(AggregateResult.class);
    }

    public PatientTasks get(String siteId, String patientId) {
        logger.debug("Getting patient tasks for a single patient.  SiteId {}, PatientId {}.", siteId, patientId);

        PatientTasks patientTasks;

        try {
            MongoCollection collection = mongo.getCollection(COLLECTION_NAME);

            patientTasks =  mongo.findOne(collection, "{patientId: '" + patientId + "', siteId: '" + siteId + "', activeVisit: true}", PatientTasks.class);
        } catch (Exception e) {
            logger.error("Unable find patient tasks.", e);
            throw new WebApp500InternalServerErrorException("Unable to find patient tasks.  Please try again later.");
        }

        if (patientTasks == null)
            patientTasks = new PatientTasks(patientId, siteId);

        return patientTasks;
    }

    public List<PatientTasks> get(String siteId, List<String> patientIds) {
        logger.debug("Getting multiple patients for site {}.", siteId);

        Iterable<PatientTasks> patientTasks;

        try {
            MongoCollection collection = mongo.getCollection(COLLECTION_NAME);

            patientTasks =  mongo.find(collection, "{patientId: {$in: #}, siteId: '#', activeVisit: true}",
                    PatientTasks.class, patientIds, siteId);
        } catch (Exception e) {
            logger.error("Unable find patient tasks for one or more patients.", e);
            patientTasks = Collections.emptyList();
        }

        return Lists.newArrayList(patientTasks);
    }

    public PatientTasks getTask(String taskId) {
        logger.debug("Getting patient task by taskId {}.", taskId);

        PatientTasks patientTasks = null;

        try {
            MongoCollection collection = mongo.getCollection(COLLECTION_NAME);

            patientTasks =  mongo.findOne(collection, "{'tasks.id': '" + taskId + "', activeVisit: true}", PatientTasks.class);
        } catch (Exception e) {
            logger.error("Unable find patient tasks for one or more patients.", e);
        }

        if (patientTasks == null)
            throw new WebApp404NotFoundException("Task with id " + taskId + " not found.");

        return patientTasks;
    }

    public void save(PatientTasks patientTasks) {
        logger.debug("Saving patient tasks.");

        try {
            MongoCollection collection = mongo.getCollection(COLLECTION_NAME);

            collection.withWriteConcern(WriteConcern.SAFE).save(patientTasks);
        } catch (Exception e) {
            logger.error("Unable save a patient's tasks.", e);
            throw new WebApp500InternalServerErrorException("Unable to save task for patient.  Please try again later.");
        }
    }

}
