package gov.va.vamf.service.shifttransition.tasks;

import gov.va.vamf.service.shifttransition.application.repositories.RepositoryFactory;
import gov.va.vamf.service.shifttransition.completedtasks.CompletedTask;
import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp500InternalServerErrorException;
import gov.va.vamf.service.shifttransition.infrastructure.security.UserServices;
import gov.va.vamf.service.shifttransition.tasks.domain.PatientTasks;
import gov.va.vamf.service.shifttransition.tasks.domain.Task;
import gov.va.vamf.service.shifttransition.tasks.domain.TaskStatus;
import gov.va.vamf.service.shifttransition.tasks.domain.time.Range;
import gov.va.vamf.service.shifttransition.tasks.representations.NewScheduledTask;
import gov.va.vamf.service.shifttransition.tasks.representations.PatientTaskSummaries;
import gov.va.vamf.service.shifttransition.tasks.representations.PatientTaskSummary;
import gov.va.vamf.service.shifttransition.tasks.representations.PatientTasksStatus;
import gov.va.vamf.service.shifttransition.tasks.representations.ScheduledTask;
import gov.va.vamf.service.shifttransition.tasks.representations.UpdatedScheduledTask;

import java.util.Date;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This is an application service used by end points (REST api) to interface with the shift transition domain
 * and repositories.  This application service is used to work with a single patients tasks both scheduled and completed.
 */
public class PatientTaskService {
    private static Logger logger = LoggerFactory.getLogger(PatientTaskService.class);

    private final RepositoryFactory repositoryFactory;

    public PatientTaskService(RepositoryFactory repositoryFactory) {
        this.repositoryFactory = repositoryFactory;
    }

    /**
     * This method is used to get the most urgent task summary for all tasks for a patient.
     * The most urgent is defined by the tasks current status.  Order of urgency is: Past Due, Due, Pending,
     * Completed, Dismissed and NA.
     *
     * @param patientId     Along with siteId used to define the patient unique key.
     * @param siteId        Along with patientId used to define the patient unique key.
     * @param startDate     Part of date range that is used to find the most urgent status for each task.
     * @param endDate       Part of date range that is used to find the most urgent status for each task.
     * @return              Returns all tasks for a patient even if they are no longer scheduled.
     */
    public PatientTaskSummaries getLatestTaskSummary(String patientId, String siteId, Date startDate, Date endDate) {
        logger.debug("Getting latest task summaries for patient {} at site {} between {} and {}.", patientId, siteId, startDate, endDate);

        PatientTasks patientTasks = repositoryFactory.getPatientTasksRepository().get(siteId, patientId);
        Range range = new Range(startDate, endDate);

        List<PatientTaskSummary> myPatientsTaskSummaries = patientTasks.getAllTasks(range);

        for (PatientTaskSummary summary : myPatientsTaskSummaries) {
            if (summary.status.equals(TaskStatus.NA.toString())) {
                logger.debug("Found task with NA status.");

                CompletedTask completedTask = repositoryFactory.getCompletedTasksRepository()
                                                .getLast(summary.taskId, startDate, endDate);

                replaceWithCompletedTask(summary, completedTask);
            }
        }

        PatientTaskSummaries patientTaskSummaries = new PatientTaskSummaries();
        patientTaskSummaries.tasks = myPatientsTaskSummaries;

        return patientTaskSummaries;
    }

    private void replaceWithCompletedTask(PatientTaskSummary summary, CompletedTask completedTask) {
        if (completedTask != null) {
            logger.debug("Replacing task with NA status with Completed task at {}.", completedTask.dateCompleted);

            summary.status = completedTask.status;
            summary.nextDueDate = completedTask.dateCompleted;
        }
    }

    /**
     * Gets detailed information about a specific patient task.
     *
     * @param taskId    Id of task to return.
     * @return          ScheduleTas will all display information for a patient task.
     */
    public ScheduledTask getTask(String taskId) {
        logger.debug("Get task with task id {}.", taskId);

        PatientTasks patientTasks = repositoryFactory.getPatientTasksRepository().getTask(taskId);

        return patientTasks.getTask(taskId);
    }

    /**
     * Saves a new scheduled task for a single patient within a single site (facility).
     *
     * @param patientId         Along with siteId used to define the patient unique key.
     * @param siteId            Along with patientId used to define the patient unique key.
     * @param newScheduledTask  A NewScheduledTask with both required and optional information used to create
     *                          a patient task.
     * @return                  Returns new saved task.
     */
    public ScheduledTask saveNew(String patientId, String siteId, NewScheduledTask newScheduledTask) {
        logger.debug("Saving new task called {} for patient {} at site {}.", newScheduledTask.name, patientId, siteId);

        newScheduledTask.validateInput();

        PatientTasksRepository repository = repositoryFactory.getPatientTasksRepository();
        PatientTasks patientTasks = repository.get(siteId, patientId);

        UserServices userServices = new UserServices();

        Task newTask = patientTasks.addNewTask(newScheduledTask, userServices.getUserName());
        repository.save(patientTasks);

        return newTask.asScheduledTask();
    }

    /**
     * Method is used to update an existing task for a specific patient.
     *
     * @param patientId             Along with siteId used to define the patient unique key.
     * @param siteId                Along with patientId used to define the patient unique key.
     * @param taskId                TaskId to for task to update.
     * @param updatedScheduledTask  UpdatedScheduledTask contains only the information that can be updated for a task.
     * @return                      Returns updated task.
     */
    public ScheduledTask saveUpdate(String patientId, String siteId, String taskId, UpdatedScheduledTask updatedScheduledTask) {
        logger.debug("Updating task {} for patient {} at site {}.", taskId, patientId, siteId);

        updatedScheduledTask.validateInput();

        PatientTasksRepository repository = repositoryFactory.getPatientTasksRepository();
        PatientTasks patientTasks = repository.get(siteId, patientId);

        UserServices userServices = new UserServices();

        Task updatedTask = patientTasks.updateTask(updatedScheduledTask, taskId, userServices.getUserName());
        repository.save(patientTasks);

        return updatedTask.asScheduledTask();
    }

    public void setNextDueDateForTaskAndSave(String siteId, String patientId, String taskId) {
        PatientTasks patientTasks = repositoryFactory.getPatientTasksRepository().get(siteId, patientId);

        if (setNextDueDate(taskId, patientTasks))
            saveNextDueDateUpdate(patientTasks);
    }

    private boolean setNextDueDate(String taskId, PatientTasks patientTasks) {
        logger.debug("Setting next due date for task with id {}.", taskId);

        boolean success = true;

        try {
            patientTasks.setNextDueDate(taskId);
        } catch (Exception ex) {
            //this technically could happen if tasks not active anymore and a new one is being used
            //i.e. task id not found error here, no rules to stop saving completed task.
            logger.warn("Could not set next due date for task {}. " + ex.getMessage(), taskId);

            success = false;
        }

        return success;
    }

    private void saveNextDueDateUpdate(PatientTasks patientTasks) {
        try {
            repositoryFactory.getPatientTasksRepository().save(patientTasks);
        } catch (Exception ex) {
            logger.error("Could not save next due date update for task.", ex);
            throw new WebApp500InternalServerErrorException("There was an error setting the next due date for this task.  " +
                                              "Please dismiss this task to advance the next due date for this task.");
        }
    }

    public List<PatientTasksStatus> getMostUrgentTaskStatusForPatients(Range range, String siteId, List<String> patientIds) {
        List<PatientTasks> myPatientTasks = repositoryFactory.getPatientTasksRepository().get(siteId, patientIds);

        TaskTransformService transformService = new TaskTransformService();
        return transformService.getMostUrgentTaskStatus(myPatientTasks, range , patientIds);
    }

    public List<PatientTaskSummary> getPatientSummariesForPatients(Range range, String siteId, List<String> patientIds) {
        List<PatientTasks> myPatientTasks = repositoryFactory.getPatientTasksRepository().get(siteId, patientIds);

        return new TaskTransformService().getTasksSummariesWithoutNAStatusTask(myPatientTasks, range);
    }
}



