package gov.va.vamf.service.shifttransition.infrastructure.patientsearch;

import com.sun.jersey.api.client.*;
import com.sun.jersey.client.urlconnection.HTTPSProperties;
import gov.va.vamf.service.shifttransition.application.config.*;
import org.codehaus.jackson.JsonNode;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.*;

import javax.net.ssl.*;
import java.security.NoSuchAlgorithmException;

/**
 * Calls pvs patient resource to determine if a patient is still admitted to the facility/ site by checking the inpatient
 * flag on the returned patient.
 */
public class InpatientResource {
    private static Logger logger = LoggerFactory.getLogger(InpatientResource.class);

    private static final Client httpClient = Client.create();
    private static final ObjectMapper mapper = new ObjectMapper();

    private final String patientSearchBaseUrl;

    public InpatientResource(ShiftTransitionConfig config) {
        patientSearchBaseUrl = config.patientSearchBaseUrl();

        httpClient.setConnectTimeout(config.patientConnectionTimeout());
        httpClient.setReadTimeout(config.patientReadTimeout());

        addHostVerify();
    }

    private void addHostVerify() {
        if (patientSearchBaseUrl.startsWith("https:")) {
            try {
                httpClient.getProperties().put(HTTPSProperties.PROPERTY_HTTPS_PROPERTIES, new HTTPSProperties(new HostnameVerifier() {
                    @Override
                    public boolean verify(String s, SSLSession sslSession) {
                        return true;
                    }
                }, SSLContext.getDefault()));
            } catch (NoSuchAlgorithmException e) {
                logger.error("Unable to create HostNameVerifier.  Cannot create https connections.", e);
                throw new RuntimeException("Unable to configure secure connection.");
            }
        }
    }

    //Default is found = true because a false would be mean the patient has been discharged
    //and we can't know know that for sure on error.  It is better to leave records in
    //the patient registry then it is to remove them when a patient is still on the facility.
    public boolean patientStillAtFacility(String siteId, String patientId, String token) {
        logger.debug("Checking to see if patient still at facility: patientId {}, siteId {}", patientId, siteId);

        String url = buildSearchUrl(siteId, patientId);
        ClientResponse response = getClientResponse(buildToken(token), url);

        if (response == null || response.getStatus() != 200)
            return true;

        return isInpatient(response);
    }

    private String buildSearchUrl(String siteId, String patientId) {
        String id = siteId.toLowerCase();

        if (!id.startsWith("dfn-"))
            id = "dfn-" + id;

        return patientSearchBaseUrl + "/" + id + "/" + patientId;
    }

    private String buildToken(String token) {
        if (token.startsWith("Bearer"))
            return token;
        else
            return "Bearer " + token;
    }

    private ClientResponse getClientResponse(String token, String url) {
        ClientResponse clientResponse;

        try {
            clientResponse = httpClient.resource(url)
                             .header("Authorization", token)
                             .accept("application/json")
                             .get(ClientResponse.class);
        } catch (Exception e) {
            logger.error("There was an error getting a response from " + url, e);
            clientResponse = null;
        }

        if (clientResponse != null && clientResponse.getStatus() != 200) {
            logger.warn("Patient get did not return an Ok response.  Code was {}.", clientResponse.getStatus());
            clientResponse = null;
        }

        return clientResponse;
    }

    private boolean isInpatient(ClientResponse response) {
        try {
            JsonNode patientNode = mapper.readTree(response.getEntityInputStream());
            return patientNode.get("inpatient").asBoolean();

        } catch (Exception e) {
            logger.warn("There was an error reading the json from the get patient response.",  e);
            return true;
        }
    }
}
