package gov.va.vamf.service.shifttransition.infrastructure.mongo;

import com.google.common.collect.Lists;
import com.mongodb.*;
import gov.va.vamf.service.shifttransition.application.config.*;
import org.jongo.*;
import org.slf4j.*;

import java.net.UnknownHostException;
import java.util.List;

/**
 * Wrapper around Jongo and Java Mongo Driver.
 */
public class MongoWrapper {
    private static final String LOGIN_ERR_MSG = "Unable able to log into shift transition db.  Make sure authentication configuration is correct. " +
            "Please check the db logs for more information.";
    private static final String DB_CONN_ERR_MSG = "Unable to connect the the mongo database.  " +
            "Please check configuration, db server(s) running and that the db server(s) can be reached from this host.";

    private static Logger logger = LoggerFactory.getLogger(MongoWrapper.class);

    private static MongoClient mongo = null;
    private static List<ServerAddress> addresses;
    private static ShiftTransitionConfig shiftTransitionConfig;
    /**
     * Supports connecting to a single mongo instance or a replica set.
     */
    public static void initializeConnection(ShiftTransitionConfig config) {
        addresses = Lists.newArrayList();
        shiftTransitionConfig = config;

        createAddresses();
        createDBConnection();
        authenticateUser();

        logger.debug("Connected to shift transition service database.");
    }

    private static void createAddresses() {
        try {
            for (Host host : shiftTransitionConfig.hosts()) {
                addresses.add(new ServerAddress(host.name, host.port));
            }
        } catch (UnknownHostException e) {
            logger.error("Cannot connection to shift transition database. Invalid host or port.", e);
            throw new RuntimeException("Cannot connection to shift transition database. Invalid host or port.", e);
        }
    }

    private static void createDBConnection() {
        if (mongo != null)
            return;

        try {
            mongo = new MongoClient(addresses);
        } catch (Exception ex) {
            logger.error(DB_CONN_ERR_MSG, ex);
            throw new RuntimeException(DB_CONN_ERR_MSG, ex);
        }
    }

    private static void authenticateUser() {
        if (mongo == null || shiftTransitionConfig.userName() == null)
            return;

        DB db = mongo.getDB("shift_transition_registry");
        boolean auth = db.authenticate(shiftTransitionConfig.userName(), shiftTransitionConfig.pwd().toCharArray());

        if (!auth) {
            logger.error(LOGIN_ERR_MSG);
            throw new RuntimeException(LOGIN_ERR_MSG);
        }
    }

    public static void closeConnection() {
        if (mongo != null)
            mongo.close();

        mongo = null;  //This is by design.  Forces a new db connection to be created on next call.
    }

    public MongoCollection getCollection(String collectionName) {
        createDBConnection();
        authenticateUser();

        try {
            DB db = mongo.getDB("shift_transition_registry");

            Jongo jongo = new Jongo(db);
            return jongo.getCollection(collectionName);
        } catch (Exception e) {
            throw handlerException("Unable to get shift transition database.", e);
        }
    }

    public <T> T findOne(MongoCollection collection, String query, Class<T> clazz) {
        try {
            return collection.findOne(query).as(clazz);
        } catch (Exception ex) {
            throw handlerException("Unable to find one for query: " + query, ex);
        }
    }

    public <T> Iterable<T> find(MongoCollection collection, String query, Class<T> clazz, Object... parameters) {
        try {
            return collection.find(query,parameters).as(clazz);
        } catch (Exception ex) {
            throw handlerException("Unable to find for query: " + query, ex);
        }

    }

    public <T> Iterable<T> findLastOne(MongoCollection collection, String query, String sortOrder, Class<T> clazz, Object... parameters) {
        try {
            return collection.find(query, parameters).sort(sortOrder).limit(1).as(clazz);
        } catch (Exception ex) {
            throw handlerException("Unable to find last for query: " + query, ex);
        }

    }

    private RuntimeException handlerException(String message, Exception ex) {
        closeConnection();
        logger.error(message, ex);
        return new RuntimeException(message, ex);
    }
}
