package gov.va.vamf.service.shifttransition.dischargeprocess;

import com.google.common.cache.*;
import com.google.common.eventbus.Subscribe;
import org.slf4j.*;

import java.util.concurrent.TimeUnit;

/**
 * This handle maintains a list of sites for which the discharge process has been executed in the last 26 hrs.
 *
 * This is a stateful handler and should not be executed asynchronously.
 */
public class SiteCacheHandler {
    private static Logger logger = LoggerFactory.getLogger(SiteCacheHandler.class);

    private static final int SITE_PROCESS_INTERVAL = 26;
    private final Cache<String, String> siteIds = CacheBuilder.newBuilder()
                                                              .expireAfterWrite(SITE_PROCESS_INTERVAL, TimeUnit.HOURS)
                                                              .build();
    private final DischargeProcessBus bus;

    public SiteCacheHandler(DischargeProcessBus bus) {
        this.bus = bus;
    }

    @Subscribe
    public void addSiteHandler(AddSiteEvent event) {
        if (siteIds.getIfPresent(event.siteId()) != null)
            return;

        logger.debug("Adding site id {} to discharge process cache.", event.siteId());

        siteIds.put(event.siteId(), event.siteId());
        bus.post(new ProcessSiteEvent(event));
    }

    @Subscribe
    public void removeSiteHandler(RemoveSiteEvent event) {
        logger.debug("Removing site id {} from cache.", event.siteId());

        siteIds.invalidate(event.siteId());
    }
}
