package gov.va.vamf.service.shifttransition.dischargeprocess;

import com.google.common.eventbus.Subscribe;
import gov.va.vamf.service.shifttransition.tasks.PatientTasksRepository;
import gov.va.vamf.service.shifttransition.tasks.domain.PatientTasks;
import gov.va.vamf.service.shifttransition.application.repositories.*;
import gov.va.vamf.service.shifttransition.infrastructure.patientsearch.InpatientResource;
import org.slf4j.*;

import java.util.*;

/**
 * Handles the discharge process for patients at single site.  Patients are first filtered by determining if any active
 * patients for a site have not had a completed task for the last 26hrs.  Only patients without any completed tasks in the
 * last 26hrs are forwarded along the discharge process.  Using the VAMF patient resource patients are filtered again.
 * If the patient's is no longer flagged as being inpatient then the patient is final remove from the patient registry.
 *
 * Removal is a 2 step process.  First the patient's tasks collection is deactivated.  Then the patient is removed from
 * all nurse watch lists.
 *
 * This handler processes one site at a time by design.  This simplifies the design and limits VAMF patient resource
 * load.  Making this handler asynchronous would require modifying the DischargeServiceBus backing Executor.  This handler
 * is thread safe as long as resource and bus dependencies remain thread safe.
 *
 * @see DischargeProcessBus
 * @see ProcessSiteEvent
 */
public class PatientDischargeHandler {
    private static Logger logger = LoggerFactory.getLogger(PatientDischargeHandler.class);

    private final RepositoryFactory repositoryFactory;
    private final InpatientResource inpatientResource;
    private final DischargeProcessBus bus;

    public PatientDischargeHandler(RepositoryFactory repositoryFactory, InpatientResource inpatientResource, DischargeProcessBus bus) {
        this.repositoryFactory = repositoryFactory;
        this.inpatientResource = inpatientResource;
        this.bus = bus;
    }

    @Subscribe
    public void processSiteDischargesHandler(ProcessSiteEvent event) {
        logger.debug("Starting discharge process for site {}.", event.siteId());

        try {
            List<AggregateResult> patientTasksResults = getAllActivePatientTasks(event.siteId());
            List<AggregateResult> completedTasksResults = getPatientsWithCompletedTaskInLast26Hours(event.siteId());

            removePatientsWithCompletedTasks(patientTasksResults, completedTasksResults);

            for (AggregateResult result : patientTasksResults) {
                removeAllDischargedPatientForSite(result, event.token());
            }
        } catch (Exception e) {
            logger.error("Error processing discharged patients for site " + event.siteId() + ". Process aborted and will start again on next user access.", e);
            removeSiteIdFromSiteCache(event.siteId());
        }

        logger.debug("Discharge process complete for site {}.", event.siteId());
    }

    private List<AggregateResult> getAllActivePatientTasks(String siteId) {
        logger.debug("Getting all active patients in patient tasks registry for site {}.", siteId);

        return repositoryFactory.getPatientTasksRepository().getAllActivePatientTasksForSite(siteId);
    }

    private List<AggregateResult> getPatientsWithCompletedTaskInLast26Hours(String siteId) {
        logger.debug("Getting list of patients with completed tasks within the last 26 hours for site {}.", siteId);

        return repositoryFactory.getCompletedTasksRepository().getPatientsWithCompletedTaskInLast26HoursForSite(siteId);
    }

    private void removePatientsWithCompletedTasks(List<AggregateResult> patientTasksResults, List<AggregateResult> completedTasksResults) {
        logger.debug("Removing patients with completed tasks from active patients.");

        for (AggregateResult completedTasksResult : completedTasksResults) {
            if (patientTasksResults.contains((completedTasksResult))) {
                AggregateResult patientTasksResult = patientTasksResults.get(patientTasksResults.indexOf(completedTasksResult));

                if (patientTasksResult != null)
                    patientTasksResult.patientIds.removeAll(completedTasksResult.patientIds);
            }
        }
    }

    private void removeAllDischargedPatientForSite(AggregateResult result, String token) {
        if (result.patientIds == null || result.patientIds.size() == 0)
            return;

        logger.debug("Removing patients for site {} if discharged.", result._id);

        for (String patientId : result.patientIds) {
            removeDischargedPatients(result._id, patientId, token);
        }
    }

    private void removeDischargedPatients(String siteId, String patientId, String token) {
        if (patientNotAtFacility(siteId, patientId, token)) {
            logger.debug("Removing patient {} for site {} from active patient tasks registry.", patientId, siteId);

            deactivatePatientTasks(siteId, patientId);
            removePatientFromAllWatchList(siteId, patientId);
        }
    }

    private boolean patientNotAtFacility(String siteId, String patientId, String token) {
        return !inpatientResource.patientStillAtFacility(siteId, patientId, token);
    }

    private void deactivatePatientTasks(String siteId, String patientId) {
        logger.debug("Deactivating patient tasks patient {} for site {}.", patientId, siteId);

        PatientTasksRepository repository = repositoryFactory.getPatientTasksRepository();

        PatientTasks patientTasks = repository.get(siteId, patientId);
        patientTasks.deactivate();

        repository.save(patientTasks);
    }

    private void removePatientFromAllWatchList(String siteId, String patientId) {
        logger.debug("Removing patient {} for site {} from all nurse watch lists.", patientId, siteId);

        repositoryFactory.getMyPatientsRepository().removePatientFromAllWatchList(siteId, patientId);
    }

    private void removeSiteIdFromSiteCache(String siteId) {
        bus.post(new RemoveSiteEvent(siteId));
    }
}
