package gov.va.vamf.service.shifttransition.dischargeprocess;

import com.google.common.eventbus.*;
import org.slf4j.*;

import java.util.concurrent.*;

/**
 * DischargeProcessBus is used to process the the removal of patients from the registry and nurse's watch lists
 * for a site.
 *
 * This class is backed by Guava's AsyncEventBus to allow events to be published asynchronously without blocking. It is
 * a thread pool of 2 because there are only 3 synchronous handlers and each handler is executed synchronously.  That
 * is each handler processes events synchronously.  Two of the handlers execute very quickly.  Should someone add any
 * more long running synchronous handlers then the thread pool should be adjusted accordingly.
 *
 * Handlers are synchronous by design to simplify there execution i.e. don't have to worry about shared state and handlers
 * can have state.  If they were executed asynchronously then the handlers would have to modified to remove and handle
 * shared state differently.
 *
 * For this process, synchronous handlers works well.
 *
 * Handlers should be added at the start of service.
 *
 * See handlers for more details.
 */
public class DischargeProcessBus {
    private static final int SHUTDOWN_WAIT_TIME = 60;

    private static Logger logger = LoggerFactory.getLogger(DischargeProcessBus.class);

    private static final ExecutorService executor = Executors.newFixedThreadPool(2);
    private static final EventBus bus = new AsyncEventBus(executor);

    public void addEventHandler(Object eventHandler) {
        bus.register(eventHandler);
    }

    public void post(Object event) {
        bus.post(event);
    }

    /**
     * Stop executor to stop handlers.  Try to let them finish before forcing them to stop.
     *
     * Process taking from Java Executor API docs.
     */
    public void stop() {
        logger.debug("Shutting down discharge process bus.");

        executor.shutdown();
        try {
            // Wait a while for existing tasks to terminate
            if (!executor.awaitTermination(SHUTDOWN_WAIT_TIME, TimeUnit.SECONDS)) {
                executor.shutdownNow();
                // Wait a while for tasks to respond to being cancelled
                if (!executor.awaitTermination(SHUTDOWN_WAIT_TIME, TimeUnit.SECONDS))
                    logger.warn("Discharge process bus did not shutdown properly.");
            }
        } catch (InterruptedException ie) {
            logger.warn("Forcing threads to shutdown.", ie);

            // (Re-)Cancel if current thread also interrupted
            executor.shutdownNow();
            // Preserve interrupt status
            Thread.currentThread().interrupt();
        }
    }
}
