package gov.va.vamf.service.shifttransition.completedtasks;

import gov.va.vamf.service.shifttransition.application.repositories.RepositoryFactory;
import gov.va.vamf.service.shifttransition.infrastructure.security.UserServices;
import gov.va.vamf.service.shifttransition.tasks.PatientTaskService;
import org.slf4j.*;

import java.util.*;

/**
 * Add a Description.  Should include class purpose (why not what) and assumptions.
 */
public class CompletedTasksService {
    private static Logger logger = LoggerFactory.getLogger(CompletedTasksService.class);

    private final RepositoryFactory repositoryFactory;

    public CompletedTasksService(RepositoryFactory repositoryFactory) {
        this.repositoryFactory = repositoryFactory;
    }

    /**
     * Save a completed task and set the next due date for the task.
     *
     * Order of execution within this method matters.  MongoDB does NOT support transactions.  Executing this method
     * advances the next due date.  If the save of the next due date fails, the user can submit a dismissed completed
     * task to advance the next due date.
     *
     * @param patientId             Along with siteId used to define the patient unique key.
     * @param siteId                Along with patientId used to define the patient unique key.
     * @param taskId                TaskId for tasks that was completed.
     * @param newCompletedTask      NewCompletedTask with information required to save a completed task.
     * @return                      Returns completed task saved if successful.
     */
    public CompletedTask saveCompleted(String patientId, String siteId, String taskId, NewCompletedTask newCompletedTask) {
        logger.debug("Saving new completed task for task with id {} for patient {} at site {}.", taskId, patientId, siteId);

        newCompletedTask.validateInput();

        CompletedTask completedTask = saveCompletedTask(patientId, siteId, taskId, newCompletedTask);

        PatientTaskService patientTaskService = new PatientTaskService(repositoryFactory);
        patientTaskService.setNextDueDateForTaskAndSave(siteId, patientId, taskId);

        return completedTask;
    }

    private CompletedTask saveCompletedTask(String patientId, String siteId, String taskId, NewCompletedTask newCompletedTask) {
        UserServices userServices = new UserServices();

        CompletedTask completedTask = create(patientId, siteId, taskId, userServices.getUserName(), newCompletedTask);

        repositoryFactory.getCompletedTasksRepository().save(completedTask);

        return completedTask;
    }

    public CompletedTask create(String patientId, String siteId, String taskId, String completedBy, NewCompletedTask newCompletedTask) {
        CompletedTask completedTask = new CompletedTask();

        completedTask.patientId = patientId;
        completedTask.siteId = siteId;
        completedTask.taskId = taskId;
        completedTask.completedBy = completedBy;
        completedTask.dateCompleted = newCompletedTask.dateCompleted == null? new Date(): newCompletedTask.dateCompleted;
        completedTask.taskName = newCompletedTask.taskName;
        completedTask.status = newCompletedTask.status;
        completedTask.comment = newCompletedTask.comment;

        return completedTask;
    }

    public List<CompletedTask> getCompletedTasksForPatients(String siteId, Date startDate, Date endDate, List<String> patientIds) {
        return repositoryFactory.getCompletedTasksRepository().get(siteId, startDate, endDate, patientIds);
    }
}
