package gov.va.vamf.service.shifttransition.completedtasks;

import gov.va.vamf.service.shifttransition.application.repositories.RepositoryFactory;
import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp400BadRequestException;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Service;

import javax.ws.rs.*;
import javax.ws.rs.core.*;

import java.util.*;

/**
 * Resource used to save and retrieve completed tasks specific to a single patient.
 */
@Service
@Scope("request")
@Path("/nsc-service/facility/{DFN-site-id}/patients/{patient-id}/completedTasks")
public class CompletedTasksResource {
    /**
     * Patient id that combine with a siteId (DFN-site-id) uniquely identifies a patient.
     */
    @PathParam("patient-id")
    private String patientId;

    /**
     * Site id that combine with a patient id uniquely identifies a patient.
     */
    @PathParam("DFN-site-id")
    private String siteId;

    private final RepositoryFactory repositoryFactory = new RepositoryFactory();

    /**
     * Get a list of completed tasks for a patient within a specific time frame.
     *
     * @param startDate Required query parameter.  The date must be an UTC timestamp.
     * @param endDate   Required query parameter.  The date must be an UTC timestamp.  This date must be after the start date.
     *
     * @return  List of CompletedTask for a patient.
     */
    @GET
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public List<CompletedTask> getCompletedTasks(@Context UriInfo info, @Context HttpHeaders headers,
                                                 @QueryParam("start-date") Long startDate, @QueryParam("end-date") Long endDate) {
    	if (null == startDate)
    		throw new WebApp400BadRequestException("start-date cannot be null");
    	if (null == endDate)
    		throw new WebApp400BadRequestException("end-date cannot be null");
        return repositoryFactory.getCompletedTasksRepository().get(siteId, new Date(startDate), new Date(endDate), patientId);
    }

    /**
     * Save new completed task information. This method should be called for all completed tasks even if
     * another resource is used to capture other task specific information like clio or orders.
     *
     * @param taskId        Task Id of the ScheduledTask that the new completed task was completed for.
     * @param completedTask NewCompletedTask that contains information to save about a task.
     *
     * @return  Completed task information if is saved successfully.  A 400 error would be returned if there is missing
     *          required data or invalid data.
     */
    @POST
    @Consumes(MediaType.APPLICATION_JSON)
    @Path("/{task-id}")
    public CompletedTask saveCompletedTask(@Context UriInfo info, @Context HttpHeaders headers,
                                      @PathParam("task-id") String taskId, NewCompletedTask completedTask) {
        CompletedTasksService service = new CompletedTasksService(repositoryFactory);
        return service.saveCompleted(patientId, siteId, taskId, completedTask);
    }

}
