package gov.va.vamf.service.shifttransition.completedtasks;

import gov.va.vamf.service.shifttransition.application.repositories.AggregateResult;
import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp500InternalServerErrorException;
import gov.va.vamf.service.shifttransition.infrastructure.mongo.MongoWrapper;

import java.util.Collections;
import java.util.Date;
import java.util.List;

import org.jongo.MongoCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;
import com.mongodb.WriteConcern;

/**
 * Interface for CompletedTasks mongodb collection.
 */
public class CompletedTasksRepository {
    public static final String COLLECTION_NAME = "patient_tasks_completed";
    public static final int TWENTY_SIX_HOURS_IN_MILLISECONDS = 93600000;
    public static final int THIRTY_MINUTES_IN_MILLISECONDS = 1800000;

    private static Logger logger = LoggerFactory.getLogger(CompletedTasksRepository.class);

    MongoWrapper mongo = new MongoWrapper();

    public List<AggregateResult> getPatientsWithCompletedTaskInLast26HoursForSite(String siteId) {
        Date now = new Date();
        Date twentySixHoursAgo = new Date(now.getTime() - TWENTY_SIX_HOURS_IN_MILLISECONDS);
        Date thirtyMinuteIntoTheFuture = new Date(now.getTime() + THIRTY_MINUTES_IN_MILLISECONDS);

        return  mongo.getCollection(COLLECTION_NAME)
                .aggregate("{$match: {siteId: #, dateCompleted: {$gte : #, $lte : #}}}", siteId, twentySixHoursAgo, thirtyMinuteIntoTheFuture)
                .and("{$group: {_id: '$siteId', patientIds: {$push: '$patientId'}}}")
                .as(AggregateResult.class);
    }

    public List<CompletedTask> get(String siteId, Date startDate, Date endDate, String patientId) {
        logger.debug("Getting completed tasks for a single patient.  SiteId {}, StartDate: {}, EndDate {}, PatientId {}.", siteId, startDate, endDate, patientId);

        return getWithParameters("{patientId: '#', siteId: '#', dateCompleted: {$gte: #, $lte: #}, status: 'Completed'}",
                                 patientId, siteId, startDate, endDate);
    }

    public List<CompletedTask> get(String siteId, Date startDate, Date endDate, List<String> patientIds) {
        logger.debug("Getting completed tasks for multiple patients.  SiteId {}, StartDate: {}, EndDate {}.", siteId, startDate, endDate);

        return getWithParameters("{patientId: {$in : #}, siteId: '#', dateCompleted: {$gte : #, $lte : #}, status: 'Completed'}",
                patientIds, siteId, startDate, endDate);
    }

    private List<CompletedTask> getWithParameters(String query, Object... parameters) {
        Iterable<CompletedTask> completedTasks;

        try {
            MongoCollection collection = mongo.getCollection(COLLECTION_NAME);

            completedTasks =  mongo.find(collection, query, CompletedTask.class, parameters);
        } catch (Exception e) {
            logger.error("Unable find completed tasks. Returning empty list.", e);
            completedTasks = Collections.emptyList();
        }

        return Lists.newArrayList(completedTasks);
    }

    /**
     * Returns the latest completed task within a given range i.e. single completed task with latest
     * completedDate.
     */
    public CompletedTask getLast(String taskId, Date startDate, Date endDate) {
        logger.debug("Getting last task by id.  TaskId {}, StartDate: {}, EndDate {}.", taskId, startDate, endDate);

        Iterable<CompletedTask> completedTasks = null;

        try {
            MongoCollection collection = mongo.getCollection(COLLECTION_NAME);

            completedTasks =  mongo.findLastOne(collection,
                    "{taskId: '#', dateCompleted: {$gte : #, $lte : #}}", "{dateCompleted : -1}",
                    CompletedTask.class, taskId, startDate, endDate);
        } catch (Exception e) {
            logger.error("Unable find last completed tasks. Returning null.", e);
        }

        if (completedTasks != null && completedTasks.iterator().hasNext())
            return completedTasks.iterator().next();
        else
            return null;
    }

    public void save(CompletedTask completedTask) {
        logger.debug("Saving new completed task. TaskId {}, PatientId {}", completedTask.taskId, completedTask.patientId);

        try {
            MongoCollection collection = mongo.getCollection(COLLECTION_NAME);

            collection.withWriteConcern(WriteConcern.SAFE).save(completedTask);
        } catch (Exception e) {
            logger.error("Unable to save a completed task.", e);
            throw new WebApp500InternalServerErrorException("Unable to save completed task for patient.  Please try again later.");
        }
    }
}

