package gov.va.vamf.service.shifttransition.application.server;

import ch.qos.logback.classic.LoggerContext;
import gov.va.vamf.service.shifttransition.application.config.ShiftTransitionConfig;
import gov.va.vamf.service.shifttransition.application.repositories.*;
import gov.va.vamf.service.shifttransition.dischargeprocess.*;
import gov.va.vamf.service.shifttransition.infrastructure.mongo.MongoWrapper;
import gov.va.vamf.service.shifttransition.infrastructure.patientsearch.InpatientResource;
import org.aeonbits.owner.ConfigFactory;
import org.slf4j.*;

import javax.servlet.*;

/**
 * Initialize application configurations and makes sure application resources are stopped properly if needed.
 *
 * Opted to initialize a few static variables in objects that use configuration setting than providing the configuration
 * either as static resource or injectable into resource objects.  No configuration was needed at the resource level.
 *
 * Added System properties to the creation of the config.  This allows you to override values using System
 * properties that match the property's key.
 *
 * @see gov.va.vamf.service.shifttransition.application.config.ShiftTransitionConfig
 * @see gov.va.vamf.service.shifttransition.infrastructure.mongo.MongoWrapper
 */
public class ServerContext implements ServletContextListener {
    private static Logger logger = LoggerFactory.getLogger(ServerContext.class);

    @Override
    public void contextInitialized(ServletContextEvent sce) {
        logger.debug("Initializing shift transition service configuration.");
        ShiftTransitionConfig config = ConfigFactory.create(ShiftTransitionConfig.class, System.getProperties());

        //config.list(System.out);

        logger.debug("Initializing shift transition service database configuration.");
        MongoWrapper.initializeConnection(config);

        new Indexes().ensureIndexes();

        logger.debug("Initializing and starting discharge process bus.");
        DischargeProcessBus bus = new DischargeProcessBus();
        bus.addEventHandler(new SiteCacheHandler(bus));
        bus.addEventHandler(new PatientDischargeHandler(new RepositoryFactory(), new InpatientResource(config), bus));
    }

    @Override
    public void contextDestroyed(ServletContextEvent sce) {
        logger.debug("Closing shift transition service database connection.");
        MongoWrapper.closeConnection();

        logger.debug("Stopping discharge process bus.");
        DischargeProcessBus bus = new DischargeProcessBus();
        bus.stop();

        logger.debug("Stopping logger context for shift transition service.");
        LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
        loggerContext.stop();
    }
}
