package gov.va.vamf.service.shifttransition.tasks;

import static com.jayway.restassured.RestAssured.baseURI;
import static com.jayway.restassured.RestAssured.port;
import static org.hamcrest.Matchers.hasItems;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertEquals;
import gov.va.vamf.service.shifttransition.ResourceTests;
import gov.va.vamf.service.shifttransition.watchlist.MyPatientsRepository;
import gov.va.vamf.service.shifttransition.tasks.PatientTasksRepository;
import gov.va.vamf.service.shifttransition.tasks.domain.TaskStatus;

import java.net.UnknownHostException;
import java.util.*;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import com.jayway.restassured.http.ContentType;
import com.jayway.restassured.response.Response;

public class PatientTasksStatusTests extends ResourceTests {

    public static final String PATIENT_ONE = "1970";
    public static final String PATIENT_TWO = "1972";
    private static String PATIENT_BASE_URI = "/NursingShiftCompanionResources/rest/v1/nsc-service/facility/52/patients";
    private static String MY_PATIENT_LIST_BASE_URI = "/NursingShiftCompanionResources/rest/v1/nsc-service/facility/52/my-patient-list/42";

    @Before
    public void init() {
        getRequestSpecWithBearerToken()
                .body("{\"identity\": {\"assigningAuthority\": \"52\", \"uniqueId\": \"1970\"}, \"name\": {\"last\": \"skywalker\", \"first\": \"luke\"}, \"gender\": \"M\", \"dob\": " + new Date().getTime() + "}").when()
                .contentType(ContentType.JSON).when()
                .post(MY_PATIENT_LIST_BASE_URI)
                .then()
                .statusCode(200);

        getRequestSpecWithBearerToken()
                .body("{\"identity\": {\"assigningAuthority\": \"52\", \"uniqueId\": \"1972\"}, \"name\": {\"last\": \"skywalker\", \"first\": \"luke\"}, \"gender\": \"M\", \"dob\": " + new Date().getTime() + "}").when()
                .contentType(ContentType.JSON).when()
                .post(MY_PATIENT_LIST_BASE_URI)
                .then()
                .statusCode(200);

        getRequestSpecWithBearerToken()
                .body("{\"name\": \"Bowling\", \"type\" : \"Custom\", \"startDate\" : \"" + createDateWithOffset(-100000) + "\"}").with()
                .contentType(ContentType.JSON).when()
                .post(PATIENT_BASE_URI + "/" + PATIENT_ONE + "/tasks")
                .then()
                .statusCode(200);

        getRequestSpecWithBearerToken()
                .body("{\"name\": \"Swimming\", \"type\" : \"Custom\", \"startDate\" : \"" + createDateWithOffset(100000) + "\"}").with()
                .contentType(ContentType.JSON).when()
                .post(PATIENT_BASE_URI + "/" + PATIENT_ONE + "/tasks")
                .then()
                .statusCode(200);

        getRequestSpecWithBearerToken()
                .body("{\"name\": \"Jogging\", \"type\" : \"Custom\", \"startDate\" : \"" + createDateWithOffset(-100000) + "\"}").with()
                .contentType(ContentType.JSON).when()
                .post(PATIENT_BASE_URI + "/" + PATIENT_TWO + "/tasks")
                .then()
                .statusCode(200);

        getRequestSpecWithBearerToken()
                .body("{\"name\": \"Climbing\", \"type\" : \"Custom\", \"startDate\" : \"" + createDateWithOffset(100000) + "\"}").with()
                .contentType(ContentType.JSON).when()
                .post(PATIENT_BASE_URI + "/" + PATIENT_TWO + "/tasks")
                .then()
                .statusCode(200);

    }

    @After
    public void cleanUp() throws UnknownHostException {
        clearCollection(MyPatientsRepository.COLLECTION_NAME);
        clearCollection(PatientTasksRepository.COLLECTION_NAME);
    }

    @SuppressWarnings("rawtypes")
	@Test
    public void getPatientTasksStatusInRange() {
        Response response = getRequestSpecWithBearerToken()
                .when()
                .queryParam("start-date", createDateWithOffset(-200000))
                .queryParam("end-date", createDateWithOffset(200000))
                .get(MY_PATIENT_LIST_BASE_URI + "/patients/tasks/status")
                .then()
                .body("$.size()", is(2))
                .body("status", hasItems("Past Due"))
                .body("patientId", hasItems(PATIENT_ONE, PATIENT_TWO))
                .extract().response();

        ArrayList links = response.path("link");  //two sets of links, one for each patient tasks status returned
        links = (ArrayList) links.get(0);         //two links per set

        assertEquals(3, links.size());

        verifyLink(links.get(0), "get-patient-task-summaries", baseURI + ":" + port + PATIENT_BASE_URI + "/" + PATIENT_ONE + "/tasks");
        verifyLink(links.get(1), "remove-patient-my-list", baseURI + ":" + port +  MY_PATIENT_LIST_BASE_URI + "/patients/" + PATIENT_ONE);
        verifyLink(links.get(2), "get-completed-tasks", baseURI + ":" + port + PATIENT_BASE_URI + "/" + PATIENT_ONE + "/completedTasks");
    }

    @Test
    public void getPatientTasksStatusOutOfRange() {
        getRequestSpecWithBearerToken()
                .when()
                .queryParam("start-date", createDateWithOffset(2000000))
                .queryParam("end-date", createDateWithOffset(40000000))
                .get(MY_PATIENT_LIST_BASE_URI + "/patients/tasks/status")
                .then()
                .body("$.size()", is(2))
                .body("status", hasItems(TaskStatus.NA.toString()))
                .body("patientId", hasItems(PATIENT_ONE, PATIENT_TWO));
    }
}
