package gov.va.vamf.service.shifttransition.tasks;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import com.google.common.collect.Lists;

import gov.va.vamf.service.shifttransition.RepositoryTests;
import gov.va.vamf.service.shifttransition.tasks.domain.PatientTasks;
import gov.va.vamf.service.shifttransition.tasks.domain.TaskStatus;
import gov.va.vamf.service.shifttransition.tasks.domain.time.Range;
import gov.va.vamf.service.shifttransition.application.repositories.AggregateResult;
import gov.va.vamf.service.shifttransition.application.repositories.RepositoryFactory;
import gov.va.vamf.service.shifttransition.infrastructure.mongo.MongoWrapper;

import java.util.Date;
import java.util.List;

import javax.ws.rs.WebApplicationException;

import gov.va.vamf.service.shifttransition.tasks.representations.*;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

public class PatientTasksRepositoryTests extends RepositoryTests {

    private RepositoryFactory factory = new RepositoryFactory();

    @Before
    public void init() {
        PatientTasksRepository repository = factory.getPatientTasksRepository();
        PatientTasks patientTasks = repository.get("51", "1970");

        NewScheduledTask newScheduledTask = createNewScheduledTask();

        patientTasks.addNewTask(newScheduledTask, "The Dude");
        repository.save(patientTasks);

        patientTasks = repository.get("51", "1972");

        patientTasks.addNewTask(newScheduledTask, "PII");
        repository.save(patientTasks);
    }

    @After
    public void cleanup() {
    	MongoWrapper mongo = new MongoWrapper();
    	mongo.getCollection(PatientTasksRepository.COLLECTION_NAME).remove();
    }

    @Test
    public void getNewPatientTasks() {
        PatientTasksRepository repository = factory.getPatientTasksRepository();
        PatientTasks patientTasks = repository.get("51", "1999");

        assertNotNull(patientTasks);
        assertEquals(0, patientTasks.getAllTasks(getMaxRange()).size());

        PatientTasksStatus tasksStatus = patientTasks.getMostUrgentTask(getMaxRange());

        assertEquals("1999", tasksStatus.patientId);
        assertEquals(TaskStatus.NA.toString(), tasksStatus.status);
    }

    @Test
    public void getSinglePatientTasks() {
        PatientTasksRepository repository = factory.getPatientTasksRepository();
        PatientTasks patientTasks = repository.get("51", "1970");

        PatientTasksStatus tasksStatus = patientTasks.getMostUrgentTask(getMaxRange());

        assertEquals("1970", tasksStatus.patientId);
        assertNotEquals(TaskStatus.NA.toString(), tasksStatus.status);
        assertNotNull(tasksStatus.nextDueDate);
    }

    @Test
    public void getMultiplePatientTasks() {
        PatientTasksRepository repository = factory.getPatientTasksRepository();
        List<PatientTasks> patientsTasks = repository.get("51", Lists.newArrayList("1970", "1972", "1999"));

        assertEquals(2, patientsTasks.size());
    }

    @Test(expected = WebApplicationException.class)
    public void getTaskNotFound() {
        PatientTasksRepository repository = factory.getPatientTasksRepository();
        repository.getTask("123");
    }

    @Test
    public void getTask() {
        PatientTasksRepository repository = factory.getPatientTasksRepository();
        PatientTasks patientTasks = repository.get("51", "1970");

        List<PatientTaskSummary> summaries = patientTasks.getAllTasks(getMaxRange());
        String taskId = summaries.get(0).taskId;

        patientTasks = repository.getTask(taskId);
        ScheduledTask scheduledTask = patientTasks.getTask(taskId);

        assertEquals(taskId, scheduledTask.taskId);
        assertEquals("The Dude", scheduledTask.createdBy);
        assertEquals("Bowling", scheduledTask.name);
        assertEquals("Game", scheduledTask.type);
    }

    @Test
    public void deactivatedPatientTasksNotFound() {
        PatientTasksRepository repository = factory.getPatientTasksRepository();
        PatientTasks patientTasks = repository.get("51", "1970");
        patientTasks.deactivate();

        repository.save(patientTasks);

        //new one returned without tasks
        patientTasks = repository.get("51", "1970");

        assertEquals(0, patientTasks.getAllTasks(getMaxRange()).size());
    }

    @Test
    public void getAllActivePatientTasksForSite() {
        NewScheduledTask newScheduledTask = createNewScheduledTask();
        PatientTasksRepository repository = factory.getPatientTasksRepository();

        //Add patient to different site
        PatientTasks patientTasks = repository.get("52", "1970");
        patientTasks.addNewTask(newScheduledTask, "The Dude");
        repository.save(patientTasks);

        List<AggregateResult> results = repository.getAllActivePatientTasksForSite("52");

        assertEquals(1, results.size());
        assertTrue(results.get(0).patientIds.contains("1970"));
        assertFalse(results.get(0).patientIds.contains("1972"));
    }

    private NewScheduledTask createNewScheduledTask() {
        NewScheduledTask newScheduledTask = new NewScheduledTask();
        newScheduledTask.name = "Bowling";
        newScheduledTask.type = "Game";
        newScheduledTask.startDate = new Date();

        return newScheduledTask;
    }


    private Range getMaxRange() {
        return new Range(new Date(0), new Date(Long.MAX_VALUE));
    }
}
