package gov.va.vamf.service.shifttransition.tasks;

import static com.jayway.restassured.RestAssured.baseURI;
import static com.jayway.restassured.RestAssured.port;
import static org.hamcrest.Matchers.hasItems;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertEquals;
import gov.va.vamf.service.shifttransition.ResourceTests;
import gov.va.vamf.service.shifttransition.completedtasks.CompletedTasksRepository;
import gov.va.vamf.service.shifttransition.watchlist.MyPatientsRepository;
import gov.va.vamf.service.shifttransition.tasks.PatientTasksRepository;
import gov.va.vamf.service.shifttransition.tasks.domain.TaskStatus;

import java.net.UnknownHostException;
import java.util.*;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import com.jayway.restassured.http.ContentType;
import com.jayway.restassured.response.Response;


public class PatientTaskSummaryTests extends ResourceTests {
    public static final String PATIENT_ONE = "1970";
    public static final String PATIENT_TWO = "1972";
    private static String PATIENT_BASE_URI = "/NursingShiftCompanionResources/rest/v1/nsc-service/facility/52/patients";
    private static String MY_PATIENT_LIST_BASE_URI = "/NursingShiftCompanionResources/rest/v1/nsc-service/facility/52/my-patient-list/42";

    @Before
    public void init() {
        getRequestSpecWithBearerToken()
                .body("{\"identity\": {\"assigningAuthority\": \"52\", \"uniqueId\": \"1970\"}, \"name\": {\"last\": \"skywalker\", \"first\": \"luke\"}, \"gender\": \"M\", \"dob\": " + new Date().getTime() + "}").when()
                .contentType(ContentType.JSON).when()
                .post(MY_PATIENT_LIST_BASE_URI)
                .then()
                .statusCode(200);

        getRequestSpecWithBearerToken()
                .body("{\"identity\": {\"assigningAuthority\": \"52\", \"uniqueId\": \"1972\"}, \"name\": {\"last\": \"skywalker\", \"first\": \"luke\"}, \"gender\": \"M\", \"dob\": " + new Date().getTime() + "}").when()
                .contentType(ContentType.JSON).when()
                .post(MY_PATIENT_LIST_BASE_URI)
                .then()
                .statusCode(200);

        getRequestSpecWithBearerToken()
                .body("{\"name\": \"Bowling\", \"type\" : \"Custom\", \"startDate\" : \"" + createDateWithOffset(-100000) + "\"}").with()
                .contentType(ContentType.JSON).when()
                .post(PATIENT_BASE_URI + "/" + PATIENT_ONE + "/tasks")
                .then()
                .statusCode(200);

        getRequestSpecWithBearerToken()
                .body("{\"name\": \"Swimming\", \"type\" : \"Custom\", \"startDate\" : \"" + createDateWithOffset(100000) + "\"}").with()
                .contentType(ContentType.JSON).when()
                .post(PATIENT_BASE_URI + "/" + PATIENT_ONE + "/tasks")
                .then()
                .statusCode(200);

        getRequestSpecWithBearerToken()
                .body("{\"name\": \"Jogging\", \"type\" : \"Custom\", \"startDate\" : \"" + createDateWithOffset(-100000) + "\"}").with()
                .contentType(ContentType.JSON).when()
                .post(PATIENT_BASE_URI + "/" + PATIENT_TWO + "/tasks")
                .then()
                .statusCode(200);

        getRequestSpecWithBearerToken()
                .body("{\"name\": \"Climbing\", \"type\" : \"Custom\", \"startDate\" : \"" + createDateWithOffset(100000) + "\"}").with()
                .contentType(ContentType.JSON).when()
                .post(PATIENT_BASE_URI + "/" + PATIENT_TWO + "/tasks")
                .then()
                .statusCode(200);
    }

    @After
    public void cleanUp() throws UnknownHostException {
        clearCollection(MyPatientsRepository.COLLECTION_NAME);
        clearCollection(PatientTasksRepository.COLLECTION_NAME);
        clearCollection(CompletedTasksRepository.COLLECTION_NAME);
    }

    @SuppressWarnings("rawtypes")
	@Test
    public void getMultiplePatientTaskSummariesInRange() {
        Response response = getRequestSpecWithBearerToken()
                .when()
                .queryParam("start-date", createDateWithOffset(-200000))
                .queryParam("end-date", createDateWithOffset(200000))
                .get(MY_PATIENT_LIST_BASE_URI + "/patients/tasks")
                .then()
                .body("$.size()", is(4))
                .body("status", hasItems("Past Due", "Due"))
                .body("patientId", hasItems(PATIENT_ONE, PATIENT_TWO))
                .body("taskName", hasItems("Jogging", "Climbing", "Swimming", "Bowling"))
                .extract().response();

        ArrayList links = response.path("link");  //two sets of links, one for each patient tasks status returned
        links = (ArrayList) links.get(0);         //two links per set

        assertEquals(2, links.size());

        String uri = baseURI + ":" + port + PATIENT_BASE_URI;
        String taskId = response.path("get(0).taskId");

        verifyLink(links.get(0), "get-patient-task", uri + "/" + PATIENT_ONE + "/tasks/" + taskId);
        verifyLink(links.get(1), "complete-patient-task", uri + "/" + PATIENT_ONE + "/completedTasks/" + taskId);
    }

    @Test
    public void getMultiplePatientTaskSummariesOutOfRange() {
        getRequestSpecWithBearerToken()
                .when()
                .queryParam("start-date", createDateWithOffset(2000000))
                .queryParam("end-date", createDateWithOffset(40000000))
                .get(MY_PATIENT_LIST_BASE_URI + "/patients/tasks")
                .then()
                .body("$.size()", is(0));
    }


    @SuppressWarnings("rawtypes")
	@Test
    public void getSinglePatientTaskSummariesInRange() {
        Response response = getRequestSpecWithBearerToken()
                .when()
                .queryParam("start-date", createDateWithOffset(-200000))
                .queryParam("end-date", createDateWithOffset(200000))
                .get(PATIENT_BASE_URI + "/" + PATIENT_ONE + "/tasks")
                .then()
                .body("tasks.size()", is(2))
                .body("tasks.status", hasItems("Past Due", "Due"))
                .body("tasks.patientId", hasItems(PATIENT_ONE))
                .body("tasks.taskName", hasItems("Swimming", "Bowling"))
                .extract().response();

        ArrayList links = response.path("link");
        assertEquals(3, links.size());

        String uri = baseURI + ":" + port + PATIENT_BASE_URI + "/" + PATIENT_ONE;

        verifyLink(links.get(0), "get-patient-task-summaries", uri + "/tasks");
        verifyLink(links.get(1), "add-new-task", uri + "/tasks");
        verifyLink(links.get(2), "get-completed-tasks", uri + "/completedTasks");
    }

    @Test
    public void getSinglePatientTaskSummariesOutOfRange() {
        getRequestSpecWithBearerToken()
                .when()
                .queryParam("start-date", createDateWithOffset(2000000))
                .queryParam("end-date", createDateWithOffset(40000000))
                .get(PATIENT_BASE_URI + "/" + PATIENT_ONE + "/tasks")
                .then()
                .body("tasks.size()", is(2))
                .body("tasks.status", hasItems(TaskStatus.NA.toString()));
    }

    @Test
    public void getMultiplePatientTaskSummariesInRangeWithCompleted() {
        addCompletedTask();

        getRequestSpecWithBearerToken()
                .when()
                .queryParam("start-date", createDateWithOffset(-200000))
                .queryParam("end-date", createDateWithOffset(200000))
                .get(MY_PATIENT_LIST_BASE_URI + "/patients/tasks")
                .then()
                .body("$.size()", is(4))
                .body("status", hasItems("Past Due", "Due", "Completed"))
                .body("patientId", hasItems(PATIENT_ONE, PATIENT_TWO))
                .body("taskName", hasItems("Jogging", "Climbing", "Swimming", "Bowling"));
    }

    @Test
    public void getSinglePatientTaskSummariesInRangeCompletedReplaceNATask() {
        addCompletedTask();

        getRequestSpecWithBearerToken()
                .when()
                .queryParam("start-date", createDateWithOffset(-200000))
                .queryParam("end-date", createDateWithOffset(0))
                .get(PATIENT_BASE_URI + "/" + PATIENT_ONE + "/tasks")
                .then()
                .body("tasks.size()", is(2))
                .body("tasks.status", hasItems("Completed", TaskStatus.NA.toString()))
                .body("tasks.patientId", hasItems(PATIENT_ONE))
                .body("tasks.taskName", hasItems("Bowling", "Swimming"));
    }

    private void addCompletedTask() {
        Response response = getRequestSpecWithBearerToken()
                .when()
                .queryParam("start-date", createDateWithOffset(-200000))
                .queryParam("end-date", createDateWithOffset(200000))
                .get(MY_PATIENT_LIST_BASE_URI + "/patients/tasks")
                .then()
                .extract().response();

        String taskId = response.path("get(0).taskId");

        getRequestSpecWithBearerToken()
                .body("{\"taskName\": \"Bowling\", \"status\" : \"Completed\"}").with().contentType(ContentType.JSON).when()
                .post(PATIENT_BASE_URI + "/" + PATIENT_ONE + "/completedTasks/" + taskId)
                .then()
                .statusCode(200);
    }
}
