package gov.va.vamf.service.shifttransition.completedtasks;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import com.google.common.collect.Lists;
import gov.va.vamf.service.shifttransition.application.repositories.AggregateResult;
import gov.va.vamf.service.shifttransition.application.repositories.RepositoryFactory;
import gov.va.vamf.service.shifttransition.infrastructure.mongo.MongoWrapper;

import java.util.Date;
import java.util.List;
import java.util.UUID;

import gov.va.vamf.service.shifttransition.RepositoryTests;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

public class CompleteTasksRepositoryTests extends RepositoryTests {
    public static final int TEN_MINUTES = 600000;

    private RepositoryFactory factory = new RepositoryFactory();
    private String taskId;

    @Before
    public void init() {
        taskId = UUID.randomUUID().toString();

        CompletedTask completedTask = new CompletedTask();
        completedTask.completedBy = "The Dude";
        completedTask.dateCompleted = nowPlusMilliseconds(-40000);
        completedTask.status = "Completed";
        completedTask.taskId = taskId;
        completedTask.patientId = "1970";

        saveCompletedTask(completedTask);

        completedTask = new CompletedTask();
        completedTask.completedBy = "The Dude";
        completedTask.dateCompleted = new Date();
        completedTask.status = "Dismissed";
        completedTask.patientId = "1970";
        completedTask.taskId = taskId;

        saveCompletedTask(completedTask);

        completedTask = new CompletedTask();
        completedTask.completedBy = "PII";
        completedTask.dateCompleted = new Date();
        completedTask.patientId = "1972";
        completedTask.taskId = UUID.randomUUID().toString();
        completedTask.status = "Dismissed";

        saveCompletedTask(completedTask);
    }

    private void saveCompletedTask(CompletedTask completedTask) {
        completedTask.siteId = "42";
        completedTask.taskName = "Bowling";

        factory.getCompletedTasksRepository().save(completedTask);
    }

    @After
    public void cleanup() {
    	MongoWrapper mongo = new MongoWrapper();
    	mongo.getCollection(CompletedTasksRepository.COLLECTION_NAME).remove();
    }

    @Test
    public void patientHasNoCompletedTaskInRange() {
        CompletedTasksRepository repository = factory.getCompletedTasksRepository();
        List<CompletedTask> completedTasks = repository.get("42", new Date(), nowPlusMilliseconds(TEN_MINUTES), "1972");

        assertEquals(0, completedTasks.size());
    }

    @Test
    public void getCompletedTasksForSinglePatient() {
        CompletedTasksRepository repository = factory.getCompletedTasksRepository();
        List<CompletedTask> completedTasks = repository.get("42", nowPlusMilliseconds(-TEN_MINUTES), nowPlusMilliseconds(TEN_MINUTES), "1970");

        assertEquals(1, completedTasks.size());
        assertEquals("Completed", completedTasks.get(0).status);
        assertEquals("The Dude", completedTasks.get(0).completedBy);
    }

    @Test
    public void getCompletedTasksForMultiplePatients() {
        CompletedTasksRepository repository = factory.getCompletedTasksRepository();
        List<CompletedTask> completedTasks = repository.get("42", nowPlusMilliseconds(-TEN_MINUTES), nowPlusMilliseconds(TEN_MINUTES), Lists.newArrayList("1972", "1970"));

        assertEquals(1, completedTasks.size());
        assertEquals("Completed", completedTasks.get(0).status);
        assertEquals("The Dude", completedTasks.get(0).completedBy);
    }

    @Test
    public void noLastCompletedTaskFoundInRange() {
        CompletedTasksRepository repository = factory.getCompletedTasksRepository();
        CompletedTask completedTask = repository.getLast(taskId, nowPlusMilliseconds(40000), nowPlusMilliseconds(TEN_MINUTES));

        assertNull(completedTask);
    }

    @Test
    public void getLastCompletedTaskForPatientInRange() {
        CompletedTasksRepository repository = factory.getCompletedTasksRepository();
        CompletedTask completedTask = repository.getLast(taskId, nowPlusMilliseconds(-TEN_MINUTES), nowPlusMilliseconds(TEN_MINUTES));

        assertNotNull(completedTask);
        assertEquals("The Dude", completedTask.completedBy);
        assertEquals("Bowling", completedTask.taskName);
        assertEquals("Dismissed", completedTask.status);
    }

    @Test
    public void getPatientsWithCompletedTaskInLast26HoursForSingleSite() {
        CompletedTasksRepository repository = factory.getCompletedTasksRepository();
        List<AggregateResult> results = repository.getPatientsWithCompletedTaskInLast26HoursForSite("42");

        assertEquals(1, results.size());
        assertEquals(2, results.get(0).patientIds.size());
        assertTrue(results.get(0).patientIds.contains("1970"));
        assertTrue(results.get(0).patientIds.contains("1972"));
    }

    private Date nowPlusMilliseconds(long milliseconds) {
        return new Date(new Date().getTime() + milliseconds);
    }
}
