module.exports = function (grunt) {
	'use strict';

	var appConfig = require('./appConfig.json'),
		portal = require('./grunt/portal'),
		generatedPortalOptions;

	portal.setGrunt(grunt);
	generatedPortalOptions = portal.createRequireJSDependencyModulesAndFetchList();

	grunt.initConfig({
		appConfig: require('./appConfig.json'),
		gruntConfig: require('./grunt/gruntConfig.json'),
		requirejs: {
			portalJS: {
				options: {
					wrap: false, // wrap false for handlebars
					preserveLicenseComments: false,
					skipDirOptimize: true, // otherwise it fails trying to optimize already minified libraries
					wrapShim: true,
					optimize: 'uglify2',
					optimizeCss: 'standard',
					uglify2: {
						topLevel: true,
						mangle: false,
						beautify: true,
						reserved: 'require, exports, $, _'
					},
					//generateSourceMaps: true,
					mainConfigFile: '<%= gruntConfig.requireJs.mainConfigFile %>',
					dir: '<%= gruntConfig.requireJs.optimizationDirectory %>',
					modules: generatedPortalOptions.dynamicallyLoadedModules,
					pragmasOnSave: {
						excludeHbsParser: true,
						excludeHbs: true,
						excludeAfterBuild: true
					},
					locale: "en_us",
					hbs: {
						templateExtension: '<%= gruntConfig.hbs.templateExtension %>',
						helperDirectory: '<%= gruntConfig.hbs.helperDirectory %>',
						i18nDirectory: '<%= gruntConfig.hbs.i18nDirectory %>',
						compileOptions: {}
					}
				}
			}
		},
		copy: {
			optimizedContainer: { // copy from rjs-optimized to dist/container
				filter: 'isFile',
				expand: true,
				cwd: '<%= gruntConfig.requireJs.optimizationDirectory %>',
				src: '<%= gruntConfig.distributionPackage.requireJsOptimizationDirectoryFiles %>',
				dest: '<%= gruntConfig.distributionPackage.directory %>'
			},
			app: { // copy images, icons, version.json from app's root to dist
				filter: 'isFile',
				expand: true,
				src: '<%= gruntConfig.distributionPackage.appRootFiles %>',
				dest: '<%= gruntConfig.distributionPackage.directory %>'
			}
		},
		shell: {
			clean: {
				command: 'rm -rf container/; rm -rf build/'
			},
			unzipWarToPhonegapFolder: {
				command: 'unzip -o build/<%= appConfig["artifact-name"] %>.war -d phonegap/www'
			},
			selenium: {
				cwd: 'test',
				command: 'rspec'
			},
			publishRelease: {
				command: 'git rev-list HEAD --count',
				options: {
					callback: function (err, stdout, stderr, cb) {
						// publish release with git commit count in version
						if (!err) {
							grunt.config.set('artifactory.release.options', {
								publish: [{
									id: 'com.agilex.healthcare.portal.js:' + appConfig["artifact-name"] + ':tgz',
									version: appConfig.version + '.' + stdout.trim(),
									path: 'build/'
								}]
							});
							grunt.task.run('artifactory:release:publish');
						} else {
							console.log('Command to get Git commit count failed');
							console.log(err);
							console.log(stderr);
						}
						cb();
					}
				}
			},
			version: {
				command: 'git log -1 --pretty=format:%h',
				options: {
					callback: function log(err, stdout, stderr, cb) {
						grunt.file.write('version.json', JSON.stringify({
							"title": appConfig.title,
							"version": appConfig.version,
							"about-description": appConfig["about-description"],
							"help-description": appConfig["help-description"],
							"artifact-name": appConfig["artifact-name"],
							"uber-portal": appConfig["uber-portal"],
							"hybrid": grunt.option('isHybrid') ? true : false,
							"default-view" : appConfig['default-view'],
							"patient-view": appConfig["patient-view"],
							"staff-view": appConfig["staff-view"],
							metaRevision: stdout,
							date: grunt.template.today()
						}));
						cb();
					}
				}
			}
		},
		compress: {
			war: {
				options: {
					archive: 'build/<%= appConfig["artifact-name"] %>.war',
					mode: 'zip'
				},
				files: [{
					cwd: '<%= gruntConfig.distributionPackage.directory %>',
					expand: true,
					src: ['**/*']
				}]
			}
		},
		artifactory: {
			options: {
				url: '<%= gruntConfig.artifactory.url %>',
				repository: '<%= gruntConfig.artifactory.repository %>',
				VsID           gruntConfig.artifactory.username %>',
				VsID           gruntConfig.artifactory.password %>'
			},
			dependencies: {
				options: {
					fetch: generatedPortalOptions.fetchList
				}
			},
			snapshot: {
				files: [{
					cwd: '<%= gruntConfig.distributionPackage.directory %>',
					expand: true,
					src: ['**/*']
				}],
				options: {
					publish: [{
						id: 'com.agilex.healthcare.portal.js:<%= appConfig["artifact-name"] %>:war',
						version: '<%= appConfig["version"] %>-SNAPSHOT',
						path: 'build/'
					}]
				}
			},
			release: {
				files: [{
					cwd: '<%= gruntConfig.distributionPackage.directory %>',
					expand: true,
					src: ['**/*']
				}]
			}
		}
	});

	grunt.loadNpmTasks('grunt-contrib-requirejs');
	grunt.loadNpmTasks('grunt-artifactory-artifact');
	grunt.loadNpmTasks('grunt-shell');
	grunt.loadNpmTasks('grunt-contrib-copy');
	grunt.loadNpmTasks('grunt-contrib-compress');

	grunt.registerTask('test', ['shell:selenium']);
	grunt.registerTask('version', ['shell:version']);
	grunt.registerTask('build', ['version', 'shell:clean', 'artifactory:dependencies:fetch', 'requirejs:portalJS', 'copy', 'compress']);
	grunt.registerTask('deploy', ['build', 'artifactory:snapshot:publish']);
	grunt.registerTask('release', ['deploy', 'shell:publishRelease']);
	grunt.registerTask('default', ['build', 'watch']);

	grunt.registerTask('copyToPhoneGapProject', ['build', 'shell:unzipWarToPhonegapFolder']);
};


