package gov.va.vamf.service.clio.observation.validation;

import com.google.common.collect.Lists;
import gov.va.vamf.service.clio.application.representations.UniqueTermId;
import gov.va.vamf.service.clio.flowsheet.representations.*;
import gov.va.vamf.service.clio.observation.representations.Observation;
import org.junit.*;

import static org.junit.Assert.fail;

public class ObservationHasValidUnitTests {
    private Field field;
    private Observation observation;

    private Validator validator;

    @Before
    public void init() {
        field = new Field();
        field.dataType = "Numeric";

        Field qualifier = new Field();
        field.uniqueTermId = new UniqueTermId("1234", "2");    //unit term type = 2

        Unit pounds = new Unit();
        pounds.uniqueTermId = new UniqueTermId("1", "POUNDS");
        pounds.max = 1500;
        pounds.min = 0;
        pounds.precision = 1;

        Unit kilo = new Unit();
        kilo.uniqueTermId = new UniqueTermId("2", "KILOGRAMS");
        kilo.max = 45;
        kilo.min = 12;
        kilo.precision = 0;

        qualifier.possibleValues.add(pounds);
        qualifier.possibleValues.add(kilo);

        field.qualifiers.add(qualifier);

        observation = new Observation();
        observation.value = "200";
        observation.qualifierIds = Lists.newArrayList();
        observation.qualifierIds.add(new UniqueTermId("1", "POUNDS"));
        observation.qualifierIds.add(new UniqueTermId("2", "STANDING"));

        validator = new ObservationHasValidUnit(field, observation);
    }

    @Test
    public void observationHasValidUnit() {
        try {
            validator.validate();
        } catch (Exception e) {
            fail("Observation should be valid.");
        }
    }

    @Test
    public void fieldIsNotNumeric() {
        field.dataType = "String";

        try {
            validator.validate();
        } catch (Exception e) {
            fail("Observation should not have unit.");
        }
    }

    @Test(expected = IllegalArgumentException.class)
    public void observationDoesNotAnyQualifiers() {
        observation.qualifierIds = null;
        validator.validate();
    }

    @Test(expected = IllegalArgumentException.class)
    public void observationDoesNotContainUnit() {
        observation.qualifierIds.remove(0);
        validator.validate();
    }

    @Test(expected = IllegalArgumentException.class)
    public void observationValueHasLetter() {
        observation.value = "a23.3";
        validator.validate();
    }

    @Test(expected = IllegalArgumentException.class)
    public void observationValueHasSpecialChar() {
        observation.value = "45/3";
        validator.validate();
    }

    @Test
    public void observationValueAtMaxRange() {
        observation.value = "1500";

        try {
            validator.validate();
        } catch (Exception e) {
            fail("Observation should have valid max value.");
        }
    }

    @Test
    public void observationValueAtMinRange() {
        observation.value = "0";

        try {
            validator.validate();
        } catch (Exception e) {
            fail("Observation should have valid min value.");
        }
    }

    @Test(expected = IllegalArgumentException.class)
    public void observationValueOverMax() {
        observation.value = "1500.1";
        validator.validate();
    }

    @Test(expected = IllegalArgumentException.class)
    public void observationValueUnderMin() {
        observation.value = "-.1";
        validator.validate();
    }

    @Test
    public void observationValueAtPrecision() {
        field.dataType = "String";

        try {
            validator.validate();
        } catch (Exception e) {
            fail("Observation should not have unit.");
        }
    }

    @Test
    public void observationValueUnderPrecision() {
        field.dataType = "String";

        try {
            validator.validate();
        } catch (Exception e) {
            fail("Observation should not have unit.");
        }
    }

    @Test(expected = IllegalArgumentException.class)
    public void observationValueOverPrecision() {
        observation.value = "200.23";
        validator.validate();
    }

    @Test(expected = IllegalArgumentException.class)
    public void observationValueEndsWithDecimal() {
        observation.value = "200.";
        validator.validate();
    }

    @Test(expected = IllegalArgumentException.class)
    public void observationValueDecimalWithZeroPrecisionUnit() {
        observation.qualifierIds.clear();
        observation.qualifierIds.add(new UniqueTermId("2", "KILOGRAMS"));
        observation.value = "13.1";

        validator.validate();
    }
}
