package gov.va.vamf.service.clio.infrastructure.vista;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import gov.va.vamf.service.clio.infrastructure.vista.mdws.ClioMdwsException;

import org.junit.Before;
import org.junit.Test;

import com.agilex.soap.Element;
import com.agilex.soap.Message;

public class ClioResultRecordsTests {
    private static String EMPTY_RECORDS = "<RECORDS></RECORDS>";

    private ClioResultRecords resultRecords;
    private Message response;

    @Before
    public void init() {
        response = new Message();
        response.removeHeader();

        response.getBody()
                .addChild("executeCLIOCommandResponse", "", "http://DNS   /ClioSvc")
                .addChild("executeCLIOCommandResult")
                .addChild("text");
    }

    @Test(expected = ClioMdwsException.class)
    public void createResultRecordsWithNullMessage() {
        resultRecords = ClioResultRecords.fromSoapMessage(null);
    }

    @Test(expected = ClioMdwsException.class)
    public void createResultRecordsWithEmptyMessage() {
        resultRecords = ClioResultRecords.fromSoapMessage(new Message());
    }

    @Test
    public void createResultRecordsWithEmptyRecordList() {
        addToTextNode(EMPTY_RECORDS);
        resultRecords = ClioResultRecords.fromSoapMessage(response);

        assertFalse(resultRecords.hasNext());
    }

    @Test
    public void createResultRecordsWithEmptyNode() {
        resultRecords = ClioResultRecords.fromSoapMessage(response);

        assertFalse(resultRecords.hasNext());
    }

    @Test
    public void createResultRecordsWithNodes() {
        addToTextNode("<RECORDS><RECORD><NAME>phil</NAME></RECORD></RECORDS>");
        resultRecords = ClioResultRecords.fromSoapMessage(response);

        assertTrue(resultRecords.hasNext());

        ClioResultRecord resultRecord = resultRecords.next();

        assertEquals("phil", resultRecord.getString("NAME"));

        assertFalse(resultRecords.hasNext());
    }

    @Test()
    public void createResultRecordsWithInvalidXML() {
        addToTextNode("<RECORDS><RECORD></NAME>phil<NAME></RECORD></RECORDS>");
        resultRecords = ClioResultRecords.fromSoapMessage(response);

        assertFalse(resultRecords.hasNext());
    }

    @Test
    public void createResultRecordsWithSuccessMessage() {
        addToTextNode("1^success");
        resultRecords = ClioResultRecords.fromSoapMessage(response);

        assertFalse(resultRecords.hasNext());
    }

    @Test(expected = VistaException.class)
    public void createResultRecordsWithErrorMessage() {
        addToTextNode("-100^Failed to add observation");
        resultRecords = ClioResultRecords.fromSoapMessage(response);
    }

    @Test(expected = VistaException.class)
    public void resultsWithoutNextRecord() {
        addToTextNode(EMPTY_RECORDS);
        resultRecords = ClioResultRecords.fromSoapMessage(response);

        resultRecords.next();
    }

    private void addToTextNode(String value) {
        Element element =  response.getBody()
                .getChild(new String[]{"executeCLIOCommandResponse", "executeCLIOCommandResult", "text"}, "", "http://DNS   /ClioSvc");

        element.addElementValue("<![CDATA[" + value + "]]>");
    }
}
