package gov.va.vamf.service.clio.observation.validation;


import com.google.common.base.*;
import com.google.common.collect.FluentIterable;
import gov.va.vamf.service.clio.flowsheet.representations.Field;
import gov.va.vamf.service.clio.observation.ObservationException;
import gov.va.vamf.service.clio.observation.representations.Observation;
import org.slf4j.*;

import java.util.List;

/**
 * Verify that an observation has a matching field in a flowsheet and verify the observation.
 */
public class ObservationValidator implements Validator {
    private static Logger logger = LoggerFactory.getLogger(ObservationValidator.class);

    private final Observation observation;
    private final List<Field> fields;

    public ObservationValidator(Observation observation, List<Field> fields) {
        this.observation = observation;
        this.fields = fields;
    }

    @Override
    public void validate() {
        logger.debug("Validating observation with id {} and term id {}.", observation.id, observation.uniqueTermId.id);

        Optional<Field> fieldForObservation = findFieldFor(observation);

        if (!fieldForObservation.isPresent())
            throw new ObservationException(400, "Observation(s) not saved. An observation was provided that is not valid for this flowsheet.", "Observation id: " + observation.uniqueTermId.id);

        logger.debug("Validating observation with id {} and term id {} has valid qualifier terms.", observation.id, observation.uniqueTermId.id);
        executeValidator(new ObservationHasValidQualifierTerms(fieldForObservation.get(), observation));

        logger.debug("Validating observation with id {} and term id {} has required qualifier terms.", observation.id, observation.uniqueTermId.id);
        executeValidator(new ObservationHasRequiredQualifierTerms(fieldForObservation.get(), observation));

        logger.debug("Validating observation with id {} and term id {} has valid unit term and value.", observation.id, observation.uniqueTermId.id);
        executeValidator(new ObservationHasValidUnit(fieldForObservation.get(), observation));
    }

    private Optional<Field> findFieldFor(final Observation observation) {
        //Says: find field with matching uniqueId for the observation
        return FluentIterable.from(fields).firstMatch(new Predicate<Field>() {
            @Override
            public boolean apply(Field input) {
                return input.uniqueTermId.equals(observation.uniqueTermId);
            }
        });
    }

    private void executeValidator(Validator validator) {
        try {
            validator.validate();
        } catch (Throwable e) {
            throw new ObservationException(400, e.getMessage(), "Observation id: " + observation.uniqueTermId.id);
        }
    }
}
