package gov.va.vamf.service.clio.observation.validation;

import com.google.common.base.*;
import com.google.common.collect.*;
import com.google.common.primitives.Floats;
import gov.va.vamf.service.clio.flowsheet.representations.*;
import gov.va.vamf.service.clio.observation.representations.Observation;

/**
 * If a field is numeric then the corresponding observation must have a unit and the value must be within the clinical
 * range and the precision for the unit selected in the observation.
 *
 * Assumption: The observation contains only one qualifier id that is a unit.
 */
public class ObservationHasValidUnit implements Validator {
    private final Field field;
    private final Observation observation;

    public ObservationHasValidUnit(Field field, Observation observation) {
        this.field = field;
        this.observation = observation;
    }

    @Override
    public void validate() {
        if (!field.dataType.equalsIgnoreCase("numeric"))
            return;

        if (observation.qualifierIds == null || observation.qualifierIds.size() == 0)
            throw new IllegalArgumentException("Observation(s) not saved. Observation requires a unit. No unit found for observation " + field.displayText + ".");

        Optional<Unit> fieldUnit = getUnit();

        if (!fieldUnit.isPresent())
            throw new IllegalArgumentException("Observation(s) not saved. No matching unit found in flowsheet. No unit found for observation " + field.displayText + ".");

        Float observationValue = validateValueIsNumeric();
        validateUnitWithInRange(fieldUnit.get(), observationValue);
        validatePrecision(fieldUnit.get());
    }

    private Optional<Unit> getUnit() {
        //Says: Get all possible values in the field's list of qualifiers (transformAndConcat) and then
        //      get only the possible values that are of type Unit (filter)
        //      and find the unit that is used in the observation (firstNatch).
        return FluentIterable.from(field.qualifiers).transformAndConcat(new Function<Field, Iterable<Value>>() {
            @Override
            public Iterable<Value> apply(Field input) {
                return input.possibleValues;
            }
        }).filter(Unit.class).firstMatch(new Predicate<Unit>() {
            @Override
            public boolean apply(Unit input) {
                return observation.qualifierIds.contains(input.uniqueTermId);
            }
        });
    }

    private Float validateValueIsNumeric() {
        Float numericValue = Floats.tryParse(observation.value);

        if (numericValue == null)
            throw new IllegalArgumentException("Observation(s) not saved. Value " + observation.value + " is not a valid for observation " + field.displayText + ".");

        return numericValue;
    }

    private void validateUnitWithInRange(Unit fieldUnit, Float observationValue) {
        if (!Range.closed(fieldUnit.min, fieldUnit.max).contains(observationValue))
            throw new IllegalArgumentException("Observation(s) not saved. Value is not within the clinical min and max range (" +
                                                fieldUnit.min + " <= " + observation.value + " <= " + fieldUnit.max + ") for observation " +
                                                field.displayText + ".");
    }

    private void validatePrecision(Unit fieldUnit) {
        if (observation.value.endsWith("."))
            throw new IllegalArgumentException("Observation(s) not saved. Value " + observation.value +
                    " ends with a decimal for observation " +
                    field.displayText + ".");
        else if (fieldUnit.precision == 0 && observation.value.contains("."))
            throw new IllegalArgumentException("Observation(s) not saved. Value " + observation.value +
                    " has decimal precision but a whole number was expected for observation " +
                    field.displayText + ".");
        else if (observation.value.contains(".") && observation.value.substring(observation.value.indexOf(".")).length() -1 > fieldUnit.precision)
            throw new IllegalArgumentException("Observation(s) not saved. Value " + observation.value +
                    " has a higher precision than was expected for observation " +
                    field.displayText + ". The expected precision for this value is " + fieldUnit.precision + " decimal places.");
    }
}
