package gov.va.vamf.service.clio.observation.validation;

import com.google.common.base.*;
import com.google.common.collect.*;
import gov.va.vamf.service.clio.application.representations.UniqueTermId;
import gov.va.vamf.service.clio.flowsheet.representations.*;
import gov.va.vamf.service.clio.observation.representations.Observation;

import java.util.*;

/**
 * Verify that the observation contains only qualifier ids that are found in the field's qualifiers possible values.
 *
 * Assumption: Only one qualifier id per qualifier type (unit, method, position, etc) is provided in the observation.
 */
public class ObservationHasValidQualifierTerms implements Validator {
    private final Field field;
    private final Observation observation;

    public ObservationHasValidQualifierTerms(Field field, Observation observation) {
        this.field = field;
        this.observation = observation;
    }

    @Override
    public void validate() {
        if (observation.qualifierIds == null || observation.qualifierIds.size() == 0)
            return;

        Collection<UniqueTermId> fieldTerms = getAllQualifierValuesFromField();

        if (!fieldTerms.containsAll(observation.qualifierIds))
            throw new IllegalArgumentException("Observation(s) not saved.  One or more qualifier values is not valid for the observation " + field.displayText + ".");
    }

    private Collection<UniqueTermId> getAllQualifierValuesFromField() {
        //Says: Get all possible values from all qualifiers (transformAndConcat) then return each possible value's
        //      UniqueTermId (transform) as a list.
        return FluentIterable.from(field.qualifiers).transformAndConcat(new Function<Field, Iterable<? extends Value>>() {
            @Override
            public Iterable<? extends Value> apply(Field input) {
                return input.possibleValues;
            }
        }).transform(new Function<Value, UniqueTermId>() {
            @Override
            public UniqueTermId apply(Value input) {
                return input.uniqueTermId;
            }
        }).filter(Predicates.notNull()).toList();
    }
}
