package gov.va.vamf.service.clio.observation.validation;

import com.google.common.base.*;
import com.google.common.collect.FluentIterable;
import gov.va.vamf.service.clio.application.representations.UniqueTermId;
import gov.va.vamf.service.clio.flowsheet.representations.*;
import gov.va.vamf.service.clio.observation.representations.Observation;

import java.util.Collection;

/**
 * Verifies that observation includes a qualifier id for each mandatory qualifier.
 */
public class ObservationHasRequiredQualifierTerms implements Validator {
    private final Field field;
    private final Observation observation;

    public ObservationHasRequiredQualifierTerms(Field field, Observation observation) {
        this.field = field;
        this.observation = observation;
    }

    @Override
    public void validate() {
        Collection<UniqueTermId> fieldTerms = getDefaultQualifierValuesFromField();

        if (fieldTerms.size() == 0)
            return;

        if (observation.qualifierIds == null || observation.qualifierIds.size() == 0 || !fieldTerms.containsAll(observation.qualifierIds))
            throw new IllegalArgumentException("Observation(s) not saved.  One or more mandatory qualifier values are not included for the observation " + field.displayText + ".");
    }

    private Collection<UniqueTermId> getDefaultQualifierValuesFromField() {
        //Says: Get qualifier default value from all qualifiers if the qualifier is required (transform)
        //      then return not null (filter) values as a list.
        return FluentIterable.from(field.qualifiers).transform(new Function<Field, UniqueTermId>() {
            @Override
            public UniqueTermId apply(Field input) {
                return field.required ? input.defaultValueId : null;
            }
        }).filter(Predicates.notNull()).toList();
    }
}
