package gov.va.vamf.service.clio.observation.validation;

import com.google.common.base.*;
import com.google.common.collect.*;
import gov.va.vamf.service.clio.application.representations.UniqueTermId;
import gov.va.vamf.service.clio.flowsheet.representations.*;
import gov.va.vamf.service.clio.observation.ObservationException;
import gov.va.vamf.service.clio.observation.representations.Observation;

import java.util.*;

/**
 * Validates that all requires observations are part of the observation set.
 */
public class ObsSetIncludeAllRequiredFields implements Validator {
    private final List<Field> fields;
    private final List<Observation> observations;

    public ObsSetIncludeAllRequiredFields(List<Field> fields, List<Observation> observations) {
        this.fields = fields;
        this.observations = observations;
    }

    @Override
    public void validate() {
        Collection<UniqueTermId> fieldIds = getUniqueTermIdsFromFlowsheet();
        Collection<UniqueTermId> allObsIds = getUniqueTermIdsFromObservationSet();

        if (!allObsIds.containsAll(fieldIds))
            throw new ObservationException(400, "Observation(s) not saved. One or more required observations was not provided.", "");
    }

    private Collection<UniqueTermId> getUniqueTermIdsFromFlowsheet() {
        //Says: return uniqueTermId for all fields if the field is required
        return FluentIterable.from(fields).transform(new Function<Field, UniqueTermId>() {
            @Override
            public UniqueTermId apply(Field input) {
                return input.required ? input.uniqueTermId : null;
            }
        }).filter(Predicates.notNull()).toList();
    }

    private Collection<UniqueTermId> getUniqueTermIdsFromObservationSet() {
        //Says: return uniqueTermId for all observations
        return Collections2.transform(observations, new Function<Observation, UniqueTermId>() {
            @Override
            public UniqueTermId apply(Observation input) {
                return input.uniqueTermId;
            }
        });
    }
}
