package gov.va.vamf.service.clio.observation.resources;

import gov.va.vamf.service.clio.flowsheet.cache.FlowsheetService;
import gov.va.vamf.service.clio.flowsheet.representations.Flowsheet;
import gov.va.vamf.service.clio.infrastructure.vista.VistaServiceFactory;
import gov.va.vamf.service.clio.observation.*;
import gov.va.vamf.service.clio.observation.representations.ObservationSet;
import gov.va.vamf.service.clio.observation.validation.*;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Service;

import javax.ws.rs.*;
import javax.ws.rs.core.*;

/**
 * Resources used to save the observation results.
 */
@Service
@Scope("request")
@Path("/clio-service/{assigningAuthority}/observation")
public class ObservationResources {
    /**
     *
     /**
     * Sends a set of observations to a Vista to be be stored in CP flowsheets list of observations.
     * This operation validates the data and stores the data to Vista using MDWS.
     *
     * @param assigningAuthority    Identifies Vista where the observation set is saved.
     * @param flowsheetId           Flowsheet that was used to collection the observation set information.
     * @param observationSet        The observation set to save to Vista.
     *
     * @return  Saved observationSet if the data is valid and saved to Vista.
     *          Returns 400 bad request status if any of observation data is not valid.  Returns a 500 internal
     *          server error if the data cannot be saved.
     */
    @POST
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    @Path("/{flowsheet-id}")
    public ObservationSet saveObservationResult(@Context UriInfo info, @Context HttpHeaders headers,
                                                @PathParam("assigningAuthority") String assigningAuthority,
                                                @PathParam("flowsheet-id") String flowsheetId, ObservationSet observationSet) {
        verifyObservationSet(assigningAuthority, flowsheetId, observationSet);

        ObservationService service = new ObservationService(new VistaServiceFactory().get(assigningAuthority));
        service.save(observationSet);

        return observationSet;
    }

    private void verifyObservationSet(String assigningAuthority, String flowsheetId, ObservationSet observationSet) {
        observationSet.validate();

        FlowsheetService flowsheetService = new FlowsheetService(new VistaServiceFactory().get(assigningAuthority));
        Flowsheet flowsheet = flowsheetService.getFlowsheet(assigningAuthority, "{" + flowsheetId + "}");

        Validator validator = new ObservationSetValidator(observationSet, flowsheet);
        validator.validate();
    }
}