package gov.va.vamf.service.clio.observation.representations;

import gov.va.vamf.service.clio.application.representations.*;
import gov.va.vamf.service.clio.observation.ObservationException;
import gov.va.vamf.service.clio.observation.complex.BloodPressureHandler;

import javax.xml.bind.annotation.*;
import java.text.*;
import java.util.*;

/**
 * An observation set is a combination of flowsheet data entry values (observations) and
 * information to identify the event such as patient identifying information and
 * location of the patient when the observation was made.
 *
 * @see gov.va.vamf.service.clio.flowsheet.representations.Flowsheet
 */
@XmlRootElement(name = "observation-set", namespace = Namespace.ns)
public class ObservationSet {

    /**
     * Unique id generated by the service.
     */
    @XmlElement()
    public String id = "{" + UUID.randomUUID().toString().toUpperCase() + "}";

    /**
     * Optional comment about the entire set of observations.
     */
    @XmlElement()
    public Comment comment;

    /**
     * Date/time observations were saved/ recorded.  This value should be in local time not GMT time.  The require
     * format is: yyyy-MM-dd@HH:mm:ss.  Example: 2015-02-06@13:17:42
     */
    @XmlElement(required = true)
    public String enteredDateTime;

    /**
     * Optional unique identifier for the person that entered the observation data.  If this is not set then the service
     * will use the logged in user's identity for this field.
     */
    @XmlElement()
    public PersonIdentity enteredBy;

    /**
     * Optional Date/time observations were observed.    This value should be in local time not GMT time.  The require
     * format is: yyyy-MM-dd@HH:mm:ss.  Example: 2015-02-06@13:17:42.
     *
     * If this field is not provided by the client/ app then the enteredDate will be used for this value.
     */
    @XmlElement()
    public String observationDateTime;

    /**
     * Optional unique identifier for the person that made the observation(s).  If this is not set then the service
     * will use the logged in user's identity for this field.
     */
    @XmlElement()
    public PersonIdentity observedBy;

    /**
     * Unique identity of observed patient i.e. observations are for this patient.
     */
    @XmlElement(required = true)
    public PersonIdentity patient;

    /**
     * Location of patient when the observations were made.
     */
    @XmlElement(required = true)
    public Location location;

    /**
     * One or more observations documented by a nurse/ observer for a patient using a flowsheet data entry view.
     */
    @XmlElement(required = true)
    public List<Observation> observations;

    /**
     * Optional name of application that is saving the observations.  If this is not set then 'Mobile Clio Resources' will be used.
     */
    @XmlElement()
    public String source;

    /**
     * Optional version of the source that is saving the observations.
     */
    @XmlElement()
    public String sourceVersion;

    /**
     * Optional source application comment.
     */
    @XmlElement()
    public Comment sourceComment;

    public void validate() {
        if (enteredDateTime == null || !isValidDateFormat(enteredDateTime))
            throw new ObservationException(400, "A valid entered date/ time is required.",
                    "An entered date for the local date and time is required.  Valid format is: yyyy-MM-dd@HH:mm:ss.");

        if (observationDateTime != null && !isValidDateFormat(observationDateTime))
            throw new ObservationException(400, "A valid observation date/time is required if set.",
                    "An entered date for the local date and time is required.  Valid format is: yyyy-MM-dd@HH:mm:ss.");

        if (enteredBy != null)
            enteredBy.validate();

        if (observedBy != null)
            observedBy.validate();

        if (patient == null)
            throw new ObservationException(400, "A patient is required.", "");

        if (location == null)
            throw new ObservationException(400, "A location is required.", "");

        if (observations == null)
            throw new ObservationException(400, "At least one observation is required.", "");

        patient.validate();
        location.validate();

        for (Observation observation : observations)
            observation.validate();
    }

    private boolean isValidDateFormat(String date) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd@HH:mm:ss");
        sdf.setLenient(false);

        try {
            sdf.parse(date);
        } catch (ParseException e) {
            return false;
        }

        return true;
    }

    public void setDefaults(String userId, String userAssingingAuthority) {
        if (enteredBy == null)
            enteredBy = new PersonIdentity(userAssingingAuthority, userId);

        if (observedBy == null)
            observedBy = enteredBy;

        if (observationDateTime == null)
            observationDateTime = enteredDateTime;

        if (comment == null)
            comment = new Comment();

        if (source == null)
            source = "Mobile Clio Resources";

        if (sourceVersion == null)
            sourceVersion = "";

        if (sourceComment == null)
            sourceComment = new Comment();

        for (Observation observation : observations)
            observation.setDefaults();

        handleSpecialCases();
    }

    private void handleSpecialCases() {
        BloodPressureHandler bloodPressureHandler = new BloodPressureHandler(this);
        bloodPressureHandler.handle();
    }
}
