package gov.va.vamf.service.clio.observation.representations;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import gov.va.vamf.service.clio.application.representations.*;
import gov.va.vamf.service.clio.observation.ObservationException;

import javax.xml.bind.annotation.*;
import java.util.*;

/**
 * An observation made by a nurse/ observer.  An observation maps to a field and user entered value for a field
 * in a flowsheet.
 *
 * @see gov.va.vamf.service.clio.flowsheet.representations.Field
 * @see gov.va.vamf.service.clio.flowsheet.representations.Value
 * @see gov.va.vamf.service.clio.flowsheet.representations.Flowsheet
 */
@XmlRootElement(name = "observation", namespace = Namespace.ns)
public class Observation {

    /**
     * Unique id generated by the service.
     */
    @XmlElement()
    public String id = "{" + UUID.randomUUID().toString().toUpperCase() + "}";

    /**
     * Contains the Clio generated unique id for an observation as well as its canonical name.  This value maps
     * to a field's uniqueTermId from a flowsheet.
     */
    @XmlElement(required = true)
    public UniqueTermId uniqueTermId;

    /**
     * Value of the observation.  Numeric values must be within the clinical range of their corresponding unit of measure.
     * If the observation is a pick list then the value is the id of the pick list item selected.  0 or 1 should be used
     * for the value of a boolean observation.
     */
    @XmlElement(required = true)
    public String value;

    /**
     * An optional comment entered by the nurse/ observer about this observation only.
     */
    @XmlElement()
    public Comment comment;

    /**
     * Optional list of unique ids for qualifiers set for an observation.  Must include a qualifier value for all
     * required qualifiers.  If observation includes a Unit then the Unit select must be in this list.
     */
    @XmlElement()
    public List<UniqueTermId> qualifierIds;

    /**
     * Used by service for complex terms.
     */
    public String parentId;

    /**
     * Used by service for complex terms.
     */
    public String childOrder;

    public void validate() {
        if (uniqueTermId == null || Strings.isNullOrEmpty(value))
            throw new ObservationException(400, "Invalid observation.", "A unique term id and value are required.");

        uniqueTermId.validate();

        if (qualifierIds != null)
            for (UniqueTermId qualifierId : qualifierIds)
                qualifierId.validate();
    }

    public void setDefaults() {
        if (comment == null)
            comment = new Comment();

        if (qualifierIds == null)
            qualifierIds = Lists.newArrayList();

        parentId = Strings.nullToEmpty(parentId);
        childOrder = Strings.nullToEmpty(childOrder);
    }
}
