package gov.va.vamf.service.clio.observation.complex;

import com.google.common.base.*;
import com.google.common.collect.*;
import gov.va.vamf.service.clio.application.representations.UniqueTermId;
import gov.va.vamf.service.clio.observation.ObservationException;
import gov.va.vamf.service.clio.observation.representations.*;
import org.slf4j.*;

import java.util.Iterator;

/**
 * This handler is used when the blood pressure observation is saved.  It adds 2 additional observations to the observation set.
 * Clients send the combine systolic and diastolic value (ex. 120/70).  Vista requires this combine observations as well as
 * as an observation for the systolic value and an observation the diastolic value.
 */
public class BloodPressureHandler {
    private static Logger logger = LoggerFactory.getLogger(BloodPressureHandler.class);

    private final ObservationSet observationSet;

    public BloodPressureHandler(ObservationSet observationSet) {
        this.observationSet = observationSet;
    }

    public void handle() {
        Optional<Observation> observation = Iterables.tryFind(observationSet.observations, new Predicate<Observation>() {
            @Override
            public boolean apply(Observation input) {
                return input.uniqueTermId.id.equals("{B15F2DF6-CE99-B847-FE6B-3D5F174A2BCD}");
            }
        });

        if (!observation.isPresent())
            return;

        logger.debug("Blood pressure observation found for observation id {}.", observation.get().id);

        Observation systolicObservation = createSystolicObservation(observation.get());
        Observation diastolicObservation = createDiastolicObservation(observation.get());

        addValues(observation.get(), systolicObservation, diastolicObservation);

        observationSet.observations.add(systolicObservation);
        observationSet.observations.add(diastolicObservation);
    }

    public Observation createSystolicObservation(Observation observation) {
        logger.debug("Creating systolic pressure observation.");

        Observation systolicPressure = createPressureObservationFrom(observation);

        systolicPressure.uniqueTermId = new UniqueTermId("{D66590AE-5D02-A206-A7CD-7EFB22AE591D}", "SYSTOLIC PRESSURE");
        systolicPressure.childOrder = "1";

        return systolicPressure;
    }

    public Observation createDiastolicObservation(Observation observation) {
        logger.debug("Creating diastolic pressure observation.");

        Observation diastolicPressure = createPressureObservationFrom(observation);

        diastolicPressure.uniqueTermId = new UniqueTermId("{14B24DAF-A0C0-060C-B1C0-ECEC1DF36E8E}", "DIASTOLIC PRESSURE");
        diastolicPressure.childOrder = "2";

        return diastolicPressure;
    }

    private Observation createPressureObservationFrom(Observation observation) {
        Observation pressureObservation = new Observation();
        pressureObservation.parentId = observation.id;

        pressureObservation.qualifierIds = Lists.newArrayList();

        for (UniqueTermId id : observation.qualifierIds)
            pressureObservation.qualifierIds.add(id);

        pressureObservation.qualifierIds.add(new UniqueTermId("{57CF7550-FD97-351E-DF60-56F8F68BE7C6}", "MILLIMETERS OF MERCURY"));

        pressureObservation.setDefaults();

        return pressureObservation;
    }

    public void addValues(Observation observation, Observation systolicObservation, Observation diastolicObservation) {
        logger.debug("Extracting pressure values from combined observation value {}.", observation.value);

        String systolicValue = null;
        String diastolicValue = null;

        Iterator<String> bloodPressureParts = Splitter.on("/").split(observation.value).iterator();

        if (bloodPressureParts.hasNext())
            systolicValue = bloodPressureParts.next();

        if (bloodPressureParts.hasNext())
            diastolicValue = bloodPressureParts.next();

        if (Strings.isNullOrEmpty(systolicValue) || Strings.isNullOrEmpty(diastolicValue))
            throw new ObservationException(400, "Observations can not be saved. Invalid blood pressure value provided.", "Expected format: ddd/ddd");

        systolicObservation.value = systolicValue;
        diastolicObservation.value = diastolicValue;
    }

}
