package gov.va.vamf.service.clio.observation;

import com.google.common.base.*;
import com.google.common.collect.*;
import com.google.common.util.concurrent.*;
import gov.va.vamf.service.clio.application.representations.UniqueTermId;
import gov.va.vamf.service.clio.infrastructure.security.UserServices;
import gov.va.vamf.service.clio.infrastructure.vista.*;
import gov.va.vamf.service.clio.infrastructure.vista.mdws.ClioMdwsService;
import gov.va.vamf.service.clio.observation.representations.*;
import org.slf4j.*;

import java.util.*;
import java.util.concurrent.*;

/**
 * Service class used to save an observation set to a Vista using Mdws.  Mdws handles writing to correct to the correct Vista
 * using the logged in user's mwds session id.  It is assumed that all validation of all data within the observation set
 * was completed prior to trying to the observation set.
 *
 * As of project completion, was not able to test with actual Mdws service.
 */
public class ObservationService {
    private static Logger logger = LoggerFactory.getLogger(ObservationService.class);

    private static final String ADD_SET_METHOD = "AddSet";
    private static final String ADD_OBSERVATION_METHOD = "AddObservation";
    private static final String ADD_OBSERVATION_TO_SET_METHOD = "AddObservationToSet";
    private static final String ADD_QUALIFIER_METHOD = "AddQualifier";

    private final ListeningExecutorService executorService = MoreExecutors.listeningDecorator(Executors.newFixedThreadPool(2));
    private final VistaService service;
    private Stopwatch stopwatch;


    private String failedMethod;
    private String failedObsId;

    public ObservationService(VistaService service) {
        this.service = service;
    }

    /**
     * Uses Listenable futures to write Observations in parallel.  If ListenableFuture's get fails then processing stops
     * at the failure i.e. not all ListenableFutures will execute if not started.
     *
     * @param observationSet  Data to save to Mdws for an observation set.
     */
    public void save(ObservationSet observationSet) {
        debugLogWithStartTimer(observationSet.id);

        UserServices userServices = new UserServices();

        try {
            observationSet.setDefaults(userServices.getUniqueId(), userServices.getAssigningAuthority());
            addObservationSet(observationSet);

            logger.debug("Adding observations for observation set {}", observationSet.id);
            for (Observation observation : observationSet.observations) {
                addObservation(observationSet, observation);
            }
        } catch (Throwable e) {
            throw createException(e, observationSet.observationDateTime);
        } finally {
            executorService.shutdown();
            debugLogWithElapsedTime(observationSet.id);
        }
    }

    private void addObservationSet(ObservationSet set) throws Throwable {
        try {
            service.makeClioCall(ADD_SET_METHOD, set.id, set.enteredDateTime, set.enteredBy.uniqueId, 1, set.comment.text);
        } catch (Throwable e) {
            setFailureInfo(ADD_SET_METHOD, null);
            throw e;
        }
    }

    private void addObservation(ObservationSet set, Observation observation) throws Exception {
        String method = ADD_OBSERVATION_METHOD;
        try {
            service.makeClioCall(method, observation.id, observation.parentId, "", set.location.id,
                    set.observationDateTime, set.observedBy.uniqueId, observation.uniqueTermId.id,
                    set.patient.uniqueId, 1, observation.value, set.source, set.sourceComment.text,
                    "", set.sourceVersion, set.enteredDateTime, set.enteredBy.uniqueId,
                    observation.childOrder, observation.comment.text, "");

            method = ADD_OBSERVATION_TO_SET_METHOD;
            service.makeClioCall(method, set.id, observation.id);

            method = ADD_QUALIFIER_METHOD;
            for (UniqueTermId qualifierId : observation.qualifierIds)
                service.makeClioCall(ADD_QUALIFIER_METHOD, observation.id, qualifierId.id);
        } catch (Exception e) {
            setFailureInfo(method, observation.uniqueTermId.id);
            throw e;
        }

    }

    //Capture failing method call and observation id of first failing method.  This information is used when creating and
    //throw an exception for the failure.
       private  void setFailureInfo(String failedMethod, String failedObsId) {
        if (failedMethod == null)
            return;

        this.failedMethod = failedMethod;
        this.failedObsId = failedObsId;
    }

    private void debugLogWithStartTimer(String id) {
        if (logger.isDebugEnabled()) {
            logger.debug("Saving observation set with id {}.", id);
            stopwatch = Stopwatch.createStarted();
        }
    }

    private void debugLogWithElapsedTime(String id) {
        if (logger.isDebugEnabled() && stopwatch != null) {
            stopwatch.stop();
            logger.debug("Saved completed for observation set with id " + id + " completed in " + stopwatch.elapsed(TimeUnit.MILLISECONDS) + " milliseconds.");
        }
    }

    private RuntimeException createException(Throwable e, String observationDateTime) {
        if (failedMethod.equals("AddSet"))
            return new ObservationException(500, "Failed to save observation set for the flowsheet.", e.getMessage());

        String source = "interface";

        Throwable t = Throwables.getRootCause(e);

        if (t instanceof VistaException)
            source = "clio";

        return new ObservationException(500, "Failed during the save of an observation for the flowsheet. " +
                "You will need to check Vista to validate and correct any/ all observations for this flowsheet. " +
                "Observation date/time: " + observationDateTime,
                "Observation term id:" + failedObsId + ", Source: " + source + ", Message: " + t.getMessage());
    }
}
