package gov.va.vamf.service.clio.infrastructure.vista;

import com.agilex.soap.Message;
import com.google.common.base.Strings;
import gov.va.vamf.service.clio.infrastructure.vista.mdws.ClioMdwsException;
import org.slf4j.*;
import org.w3c.dom.*;

import javax.xml.parsers.*;
import javax.xml.xpath.*;
import java.io.ByteArrayInputStream;
import java.nio.charset.Charset;
import java.util.Iterator;

/**
 * Add a Description.  Should include class purpose (why not what) and assumptions.
 */
public class ClioResultRecords implements Iterator<ClioResultRecord> {
    private static final String EMPTY_RECORDS = "<RECORDS></RECORDS>";
    private static Logger logger = LoggerFactory.getLogger(ClioResultRecords.class);

    private NodeList recordNodes;
    private int currentNode = -1;

    public static ClioResultRecords fromSoapMessage(Message response) {
        logger.debug("Extract Clio data from Soap message.");
        String recordsXML = getResultText(response);

        logger.debug("Creating clio result records from soap message.");
        return new ClioResultRecords(recordsXML);
    }

    private static String getResultText(Message response) {
        try {
            return response.getBody()
                    .getChild(new String[]{"executeCLIOCommandResponse", "executeCLIOCommandResult", "text"}, "", "http://DNS   /ClioSvc")
                    .getElementValue();
        } catch (Throwable t) {
            logger.error("Unable to read SOAP response.  SOAP response may not have expected nodes: executeCLIOCommandResponse, executeCLIOCommandResult, text.", t);
            throw new ClioMdwsException("Unable to read response from Vista.");
        }
    }

    public ClioResultRecords(String recordXML) {
        createRecordsNodes(getRecords(recordXML));
    }

    private String getRecords(String recordsXML) {
        logger.debug("Records: " + recordsXML);

        if (Strings.isNullOrEmpty(recordsXML)) {
            logger.debug("Clio message did not return any data.");
            recordsXML = EMPTY_RECORDS;
        } else if (successfulOperation(recordsXML)) {
            logger.debug("Clio Mdws operation was successful.");
            recordsXML = EMPTY_RECORDS;
        } else if (notXML(recordsXML))
            throw new VistaException("Operation did not complete successfully.  Error returned from Vista. Error: " + recordsXML);

        logger.debug("Records: " + recordsXML);

        return recordsXML;
    }

    private boolean successfulOperation(String recordsXML) {
        return recordsXML.startsWith("1^");
    }

    private boolean notXML(String recordsXML) {
        return !recordsXML.startsWith("<");
    }

    private void createRecordsNodes(String recordsXML) {
        logger.debug("Creating record list from xml.");

        DocumentBuilderFactory builderFactory = DocumentBuilderFactory.newInstance();

        try {
            DocumentBuilder builder = builderFactory.newDocumentBuilder();
            Document xmlDocument = builder.parse(new ByteArrayInputStream(recordsXML.getBytes(Charset.forName("UTF-8"))));

            XPath xPath =  XPathFactory.newInstance().newXPath();
            recordNodes = (NodeList) xPath.compile("//RECORD").evaluate(xmlDocument, XPathConstants.NODESET);
        } catch (Exception e) {
            logger.error("Error creating clio records from xml.", e);
        }
    }

    @Override
    public boolean hasNext() {
        return recordNodes != null && recordNodes.getLength() != 0 && currentNode + 1 < recordNodes.getLength() ;
    }

    @Override
    public ClioResultRecord next() {
        if (recordNodes == null || recordNodes.getLength() == 0)
            throw new VistaException("No valid value available. Calling next on null result. Should have checked hasNext() method before calling next().");

        logger.debug("Getting next clio record.");

        currentNode ++;
        return new ClioResultRecord(recordNodes.item(currentNode));
    }

    @Override
    public void remove() {}
}
