package gov.va.vamf.service.clio.flowsheet.resources;

import gov.va.vamf.service.clio.application.resources.Routes;
import gov.va.vamf.service.clio.flowsheet.cache.FlowsheetService;
import gov.va.vamf.service.clio.flowsheet.representations.*;
import gov.va.vamf.service.clio.infrastructure.vista.VistaServiceFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Service;

import javax.ws.rs.*;
import javax.ws.rs.core.*;

/**
 * Resources used to retrieve flowsheets.
 */
@Service
@Scope("request")
@Path("/clio-service/{assigning-authority}/flowsheets")
public class FlowsheetsResources {
    /**
     * Nurse's assigning authority. Must be a DFN-xxx formatted.
     */
    @PathParam("assigning-authority")
    private String assigningAuthority;

    /**
     * Gets a list of flowsheet names for display in a pick list.
     *
     * @return  All available flowsheets names for a single Vista (site) based on assigning-authority.
     */
    @GET
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public FlowsheetList getList(@Context UriInfo info, @Context HttpHeaders headers) {
        FlowsheetService service = new FlowsheetService(new VistaServiceFactory().get(assigningAuthority));
        FlowsheetList items = service.getFlowsheetList(assigningAuthority);
        items.generateLinks(new Routes(info, assigningAuthority));

        return items;
    }

    /**
     /**
     * Gets information about a flowsheet needed to display a data entry view for a single flowsheet.
     *
     * @param flowsheetId   Unique id that identifies a specific flowsheet in the Vista assoicated with the assigning-authority.
     *
     * @return  A single flowsheet's data entry information.  Returns a 404 error if the flowsheet is not found.
     */
    @GET
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    @Path("/{flowsheet-id}")
    public Flowsheet get(@Context UriInfo info, @Context HttpHeaders headers, @PathParam("flowsheet-id") String flowsheetId) {
        FlowsheetService service = new FlowsheetService(new VistaServiceFactory().get(assigningAuthority));
        Flowsheet flowsheet = service.getFlowsheet(assigningAuthority, "{" + flowsheetId + "}");
        flowsheet.generateLinks(new Routes(info, assigningAuthority));

        return flowsheet;
    }
}