package gov.va.vamf.service.clio.flowsheet.representations;

import com.google.common.base.*;
import com.google.common.collect.*;
import gov.va.vamf.service.clio.application.representations.*;
import gov.va.vamf.service.clio.application.resources.Routes;

import javax.xml.bind.annotation.*;
import java.util.*;

/**
 * Represents the data entry information from a Clio flowsheet.
 * This information can be used to create a data entry view.
 */
@XmlRootElement(name = "flowsheet", namespace = Namespace.ns)
public class Flowsheet {
    /**
     * Contains the Clio generated flowsheet unique id and flowsheet name.
     */
    @XmlElement(required = true)
    public UniqueTermId uniqueTermId;

    /**
     * Optional comment generated by the flowsheet author.
     */
    @XmlElement()
    public String comment;

    /**
     * One or more field objects.  Each field object maps to a single data entry element i.e. text box, pick list, etc.
     */
    @XmlElement(required = true)
    public List<Field> fields = Lists.newArrayList();

    /**
     * Holds list of possible locations for observation(s) using this flowsheet.
     */
    @XmlElement(required = true)
    public List<Location> possibleLocations = Lists.newArrayList();

    /**
     * Returns related resources.
     * Titles ares: get-flowsheet, save-observation-result.
     */
    @XmlElement(name = "link", required = true)
    public List<Link> links = Lists.newArrayList();

    public Flowsheet() {}

    public Flowsheet(FlowsheetItem flowsheetItem) {
        uniqueTermId = flowsheetItem.uniqueTermId;
        comment = flowsheetItem.comment;
    }

    public void generateLinks(Routes routes) {
        if (links.size() > 0)
            return;

        String id = CharMatcher.anyOf("{}").removeFrom(uniqueTermId.id);

        Link selfLink = routes.getFlowsheet(id);
        selfLink.rel = Link.self;

        links.add(selfLink);
        links.add(routes.saveResult(id));
    }

    @Override
    public String toString() {
        return Objects.toStringHelper(this).omitNullValues()
                .add("id", uniqueTermId.id)
                .add("name", uniqueTermId.name)
                .add("comment", comment)
                .add("possible values", Iterables.toString(fields))
                .toString();
    }
}