package gov.va.vamf.service.clio.flowsheet.cache.location;

import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import gov.va.vamf.service.clio.application.representations.Location;
import gov.va.vamf.service.clio.flowsheet.cache.*;
import gov.va.vamf.service.clio.flowsheet.cache.flowsheet.FlowsheetException;
import gov.va.vamf.service.clio.infrastructure.security.UserServices;
import gov.va.vamf.service.clio.infrastructure.vista.*;
import org.slf4j.*;

import java.util.*;

/**
 * Returns a list of locations for a facility.  A location is required to save an observation set.
 *
 * @see gov.va.vamf.service.clio.flowsheet.cache.ClioCache
 * @see gov.va.vamf.service.clio.flowsheet.cache.CacheKey
 * @see gov.va.vamf.service.clio.infrastructure.vista.VistaService
 *
 * @see gov.va.vamf.service.clio.application.representations.Location
 */
public class LocationsLoader implements Loader<CacheKey<String>, List<Location>> {
    private static Logger logger = LoggerFactory.getLogger(LocationsLoader.class);

    @Override
    public List<Location> load(CacheKey<String> key) {
        logger.debug("Loading locations for assigningAuthority {}.", key.value);

        List<Location> locations = Lists.newArrayList();

        ClioResultRecords resultRecords = getClioResultRecords(key.vistaService);

        while (resultRecords.hasNext()) {
            ClioResultRecord resultRecord = resultRecords.next();

            locations.add(createLocation(resultRecord));
        }

        Collections.sort(locations);

        return locations;
    }

    private ClioResultRecords getClioResultRecords(VistaService vistaService) {
        try {
            return vistaService.makeClioCall("GetFacilityLocations");
        } catch (Exception e) {
            logger.error("Unable to load locations. ", e);
            throw new FlowsheetException(500, "Cannot return flowsheet. Unable to retrieve locations for your facility.",
                     Throwables.getRootCause(e).getMessage());
        }
    }

    private Location createLocation(ClioResultRecord resultRecord) {
        Location location = new Location();

        location.id = resultRecord.getLong("LOCATION_ID");
        location.name = resultRecord.getString("LOCATION_NAME");

        return location;
    }
}
