package gov.va.vamf.service.clio.flowsheet.cache.flowsheet.handlers;

import com.google.common.base.*;
import com.google.common.collect.*;
import gov.va.vamf.service.clio.flowsheet.representations.*;

import java.util.List;

/**
 * Replace any default values with their matching possible value.  During loading the UniqueTermId.name of the
 * default value is not know so we update the default after all fields have been loaded.
 */
public class UpdateDefaultValueHandler {
    private final List<Field> fields;

    public UpdateDefaultValueHandler(List<Field> fields) {
        this.fields = fields;
    }

    public void handle() {
        List<Field> fieldsWithDefaultValues = FluentIterable.from(fields).filter(hasDefaultValue()).toList();
        List<Field> qualifiersWithDefaultValues = getAllQualifierFieldsWithDefaultValue();
        List<Field> allFieldsWithDefaultValues = Lists.newArrayList(Iterators.concat(fieldsWithDefaultValues.iterator(), qualifiersWithDefaultValues.iterator()));

        updateDefaultValue(allFieldsWithDefaultValues);
    }

    public List<Field> getAllQualifierFieldsWithDefaultValue() {
        return FluentIterable.from(fields).transformAndConcat(new Function<Field, Iterable<Field>>() {
                @Override
                public Iterable<Field> apply(Field input) {
                    return input.qualifiers;
                }
            }).filter(hasDefaultValue()).toList();
    }

    private static Predicate<Field> hasDefaultValue() {
        return new Predicate<Field>() {
            @Override
            public boolean apply(Field input) {
                return input.defaultValueId != null;
            }
        };
    }

    public void updateDefaultValue(List<Field> allFieldsWithDefaultValues) {
        for (final Field field : allFieldsWithDefaultValues) {
            Optional<Value> value = Iterables.tryFind(field.possibleValues, new Predicate<Value>() {
                @Override
                public boolean apply(Value input) {
                    return input.uniqueTermId.id.equals(field.defaultValueId.id);
                }
            });

            if (value.isPresent())
                field.defaultValueId = value.get().uniqueTermId;
        }
    }
}
