package gov.va.vamf.service.clio.flowsheet.cache.flowsheet.handlers;

import com.google.common.base.*;
import com.google.common.collect.Range;
import com.google.common.eventbus.*;
import gov.va.vamf.service.clio.flowsheet.cache.flowsheet.FlowsheetException;
import gov.va.vamf.service.clio.flowsheet.cache.flowsheet.events.*;
import gov.va.vamf.service.clio.infrastructure.security.UserServices;
import gov.va.vamf.service.clio.application.representations.UniqueTermId;
import gov.va.vamf.service.clio.flowsheet.representations.Field;
import gov.va.vamf.service.clio.infrastructure.vista.*;
import gov.va.vamf.service.clio.infrastructure.vista.mdws.ClioMdwsService;
import org.slf4j.*;

/**
 * In Clio there is an association of all field (observation) terms for a view with
 * all valid possible units and qualifier values (pick list values).
 */
public class GetFlowsheetViewFiltersHandler extends FlowsheetsHandler {
    private static Logger logger = LoggerFactory.getLogger(GetFlowsheetViewFiltersHandler.class);

    private static String[] qualifierNames = new String[] {"Method", "Position", "Location", "Quality", "product"};

    public GetFlowsheetViewFiltersHandler(EventBus bus, VistaService vistaService) {
        super(bus, vistaService);
    }

    @Subscribe
    public void handle(ViewFieldsFound event) {
        logger.debug("Getting view filters for view with id {}.", event.viewId);

        Optional<ClioResultRecords> result = getClioData("GetFlowsheetViewFilters", event.viewId);

        if (!result.isPresent())
            return;

        processResults(event, result.get());
    }

    public void processResults(ViewFieldsFound event, ClioResultRecords resultRecords) {
        Range<Integer> qualifierTypeRange = Range.closed(3, 7);
        Optional<Field> field = Optional.absent();

        while (resultRecords.hasNext()) {
            ClioResultRecord resultRecord = resultRecords.next();

            int type = resultRecord.getInteger("FILTER_TYPE");

            logger.debug("Getting filter information for view {} with filter type {}.", event.viewId, type);

            //type: 1 -> Observation, 2 -> unit, 3-7 -> qualifiers
            if (type == 1)
                field = getField(event, resultRecord);
            if (type == 2 && field.isPresent())
                getUnitAddToField(field.get(), resultRecord);
            else if (qualifierTypeRange.contains(type) && field.isPresent())
                getQualifierAddToField(field.get(), resultRecord, type);
         }
    }

    public Optional<Field> getField(ViewFieldsFound event, ClioResultRecord resultRecord) {
        String termId = resultRecord.getString("FILTER_TERM");

        for (Field f : event.flowsheet.fields) {
            if (f.uniqueTermId.id.equals(termId)) {
                return Optional.fromNullable(f);
            }
        }

        //Technically this should never happen.
        bus.post(new FlowsheetException(500, "Unable to retrieve flowsheet.", "Unable to find matching field in flowsheet for termId:" + termId));

        return Optional.absent();
    }

    public void getUnitAddToField(Field field, ClioResultRecord resultRecord) {
        Field unit = createQualifier(new UniqueTermId(field.uniqueTermId.id, "2"));
        unit.displayText = "Unit";
        unit.required = true;

        field.qualifiers.add(unit);

        addDefaultIfPresent(resultRecord, unit);

        bus.post(new FieldHasPossibleUnits(unit));
    }

    public void getQualifierAddToField(Field field, ClioResultRecord resultRecord, int type) {
        Field qualifier = createQualifier(new UniqueTermId(field.uniqueTermId.id, Integer.toString(type)));
        qualifier.displayText = qualifierNames[type - 3];

        field.qualifiers.add(qualifier);

        addDefaultIfPresent(resultRecord, qualifier);

        qualifier.required = resultRecord.getInteger("FILTER_USAGE") == 1;

        bus.post(new FieldHasPossibleValues(qualifier, type));
    }

    private Field createQualifier(UniqueTermId uniqueTermId) {
        Field qualifier = new Field();

        qualifier.uniqueTermId = uniqueTermId;
        qualifier.dataType = "PickList";

        return qualifier;
    }

    public void addDefaultIfPresent(ClioResultRecord resultRecord, Field field) {
        String defaultValueId = resultRecord.getString("FILTER_TERM");

        if (!Strings.isNullOrEmpty(defaultValueId))
            field.defaultValueId = new UniqueTermId(defaultValueId, "");
    }
}
