package gov.va.vamf.service.clio.flowsheet.cache.flowsheet.handlers;

import com.google.common.eventbus.*;
import gov.va.vamf.service.clio.application.representations.UniqueTermId;
import gov.va.vamf.service.clio.flowsheet.cache.flowsheet.events.*;
import gov.va.vamf.service.clio.flowsheet.representations.*;
import gov.va.vamf.service.clio.infrastructure.vista.*;
import org.slf4j.*;

/**
 * Retrieves details about a specific term from Clio.  This handle used to get details for fields only.  Fields have
 * properties that make the unique compared to possible values and units such as multi select, data type, and the display
 * name for boolean values.
 */
public class GetFieldDetailsHandler extends DetailHandler<Field> {
    private static final int PICK_LIST_TERM_TYPE = 11;
    private static Logger logger = LoggerFactory.getLogger(GetFieldDetailsHandler.class);

    public GetFieldDetailsHandler(EventBus bus, VistaService vistaService) {
        super(bus, vistaService);
    }

    @Subscribe
    public void handle(FieldFound event) {
        logger.debug("Getting field details for field with id {}.", event.field.uniqueTermId.id);

        handle(event.field);
    }

    @Override
    protected void addAdditionalData(Field value, ClioResultRecord resultRecord) {
        logger.debug("Getting additional field details for field with id {}.", value.uniqueTermId.id);

        addDataType(value, resultRecord);

        if (value.dataType.equals("PickList"))
            value.multiSelect = resultRecord.getString("MULTI_SELECT_PICKLIST").equals("1");

        if (value.dataType.equals("Boolean")) {
            addBooleanValues(value, resultRecord);
        } else if (value.dataType.equals("PickList")) {
            FieldHasPossibleValues possibleValues = new FieldHasPossibleValues(value, PICK_LIST_TERM_TYPE);
            bus.post(possibleValues);
        }
    }

    private void addBooleanValues(Field field, ClioResultRecord resultRecord) {
        Value trueValue = new Value();
        trueValue.displayText = resultRecord.getString("BOOLEAN_VALUE_TRUE");
        trueValue.uniqueTermId = new UniqueTermId("1", "YES");

        Value falseValue = new Value();
        falseValue.displayText = resultRecord.getString("BOOLEAN_VALUE_FALSE");
        falseValue.uniqueTermId = new UniqueTermId("0", "NO");

        field.possibleValues.add(trueValue);
        field.possibleValues.add(falseValue);
    }

    private void addDataType(Field field, ClioResultRecord resultRecord) {
        int dataType = resultRecord.getInteger("DATA_TYPE");

        switch (dataType) {
            case 2:
                field.dataType = "Numeric";
                break;
            case 3:
                field.dataType = "PickList";
                break;
            case 4:
                field.dataType = "Boolean";
                break;
            default:
                field.dataType = "String";
        }
    }
}
