package gov.va.vamf.service.clio.flowsheet.cache.flowsheet.handlers;

import com.google.common.base.*;
import com.google.common.eventbus.EventBus;
import gov.va.vamf.service.clio.application.representations.UniqueTermId;
import gov.va.vamf.service.clio.flowsheet.cache.flowsheet.FlowsheetException;
import gov.va.vamf.service.clio.flowsheet.representations.*;
import gov.va.vamf.service.clio.infrastructure.security.UserServices;
import gov.va.vamf.service.clio.infrastructure.vista.*;
import gov.va.vamf.service.clio.infrastructure.vista.mdws.ClioMdwsService;

/**
 * Add a Description.  Should include class purpose (why not what) and assumptions.
 */
public abstract class DetailHandler<T extends Value> extends FlowsheetsHandler {
    public DetailHandler(EventBus bus, VistaService vistaService) {
        super(bus, vistaService);
    }

    public void handle(T value) {
        Optional<ClioResultRecords> result = getClioData("GetTermByID", value.uniqueTermId.id);

        if (!result.isPresent())
            return;

        ClioResultRecords resultRecords = result.get();

        if (resultRecords.hasNext()) {
            ClioResultRecord resultRecord = resultRecords.next();

            replaceUniqueId(value, resultRecord);

            addDisplayText(value, resultRecord);
            addHelpText(value, resultRecord);
            addDescription(value, resultRecord);

            addAdditionalData(value, resultRecord);
        } else {
            bus.post(new FlowsheetException(500, "Unable to retrieve flowsheet.", "Details for term with id " + value.uniqueTermId.id + " were not found."));
        }
    }

    protected void replaceUniqueId(Value value, ClioResultRecord resultRecord) {
        value.uniqueTermId = new UniqueTermId(resultRecord.getString("ID"), resultRecord.getString("TERM"));
    }

    private void addDisplayText(Value value, ClioResultRecord resultRecord) {
        if (Strings.isNullOrEmpty(value.displayText))
            value.displayText = resultRecord.getString("DISPLAY_NAME");
    }

    private void addHelpText(Value value, ClioResultRecord resultRecord) {
        String helpText = resultRecord.getString("HELP_TEXT");

        if (helpText != null && !CharMatcher.JAVA_LETTER.retainFrom(helpText).equalsIgnoreCase("NoHelpText"))
            value.helpText = helpText;
    }

    private void addDescription(Value value, ClioResultRecord resultRecord) {
        String description = resultRecord.getString("DESCRIPTION");

        if (description != null && !CharMatcher.JAVA_LETTER.retainFrom(description).equalsIgnoreCase("NoDescription"))
            value.description = description;
    }

    protected abstract void addAdditionalData(T value, ClioResultRecord clioResultRecord);
}
