package gov.va.vamf.service.clio.flowsheet.cache.flowsheet;

import com.google.common.base.*;
import com.google.common.collect.Lists;
import com.google.common.eventbus.*;
import gov.va.vamf.service.clio.flowsheet.cache.*;
import gov.va.vamf.service.clio.flowsheet.representations.*;
import gov.va.vamf.service.clio.flowsheet.cache.flowsheet.handlers.*;
import gov.va.vamf.service.clio.infrastructure.vista.VistaService;
import org.slf4j.*;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * This is the loader for a single flowsheet.  It use Guava's EventBus to dynamically create the required pipeline to
 * retrieve all of the data required for a flowsheet.  The EventBus is configured to execute all handlers synchronously.
 * The EventBus pipeline references and passes data for a single flowsheet.  The loader creates and uses a new EventBus
 * for each load (ie thread safe).
 *
 */
public class FlowsheetRetrievalExecutive implements Loader<CacheKey<FlowsheetItem>, Flowsheet> {
    private static Logger logger = LoggerFactory.getLogger(FlowsheetRetrievalExecutive.class);

    @Override
    public Flowsheet load(CacheKey<FlowsheetItem> key) {
        EventBus bus = new EventBus();
        List<Object> handlers = Lists.newArrayList();

        Optional<Stopwatch> stopwatch = debugLogWithStartTimer(key.value.uniqueTermId.id);

        Flowsheet flowsheet = new Flowsheet(key.value);

        addHandlersToBus(bus, handlers, key.vistaService);
        ExceptionHandler exceptionHandler = addExceptionHandler(bus, handlers);

        bus.post(new FlowsheetRequestedCommand(flowsheet));

        unregisterHandlers(bus, handlers);

        if (exceptionHandler.processingException.isPresent()) {
            Exception e = exceptionHandler.processingException.get();
            throw new FlowsheetException(500, "Unable to retrieve flowsheet.",
                    Throwables.getRootCause(e).getMessage());
        }

        new UpdateDefaultValueHandler(flowsheet.fields).handle();

        debugLogWithElapsedTime(flowsheet, stopwatch);

        return flowsheet;
    }

    public Optional<Stopwatch> debugLogWithStartTimer(String id) {
        Optional<Stopwatch> stopwatch = Optional.absent();

        if (logger.isDebugEnabled()) {
            logger.debug("Getting flowsheet with id {}.", id);
            stopwatch = Optional.fromNullable(Stopwatch.createStarted());
        }

        return stopwatch;
    }

    public void debugLogWithElapsedTime(Flowsheet flowsheet, Optional<Stopwatch> stopwatch) {
        if (logger.isDebugEnabled() && stopwatch.isPresent()) {
            stopwatch.get().stop();
            logger.debug("Flowsheet retrieved for " + flowsheet.uniqueTermId.id + " completed in " + stopwatch.get().elapsed(TimeUnit.MILLISECONDS) + " milliseconds.");
            logger.debug(flowsheet.toString());
        }
    }

    private void addHandlersToBus(EventBus bus, List<Object> handlers, VistaService vistaService) {
        handlers.add(new GetFlowsheetViewsHandler(bus, vistaService));
        handlers.add(new GetFlowsheetViewTermsHandler(bus, vistaService));
        handlers.add(new GetFlowsheetViewFiltersHandler(bus, vistaService));
        handlers.add(new GetPossibleValuesHandler(bus, vistaService));
        handlers.add(new GetValueDetailsHandler(bus, vistaService));
        handlers.add(new GetFieldDetailsHandler(bus, vistaService));
        handlers.add(new GetUnitsHandler(bus, vistaService));

        for (Object handler : handlers)
            bus.register(handler);
    }

    private ExceptionHandler addExceptionHandler(EventBus bus, List<Object> handlers) {
        ExceptionHandler exceptionHandler = new ExceptionHandler();

        bus.register(exceptionHandler);
        handlers.add(exceptionHandler);

        return exceptionHandler;
    }

    private void unregisterHandlers(EventBus bus, List<Object> handlers) {
        for (Object handler : handlers)
            bus.unregister(handler);

        handlers.clear();
    }

    private static class ExceptionHandler {
        private Optional<Exception> processingException = Optional.absent();

        @Subscribe
        public void handleException(Exception exception) {
            if (processingException.isPresent())
                return;

            processingException = Optional.fromNullable(exception);

            logger.error("Exception retrieving flowsheet.", exception);
        }

    }
}
