package gov.va.vamf.service.clio.flowsheet.cache.flowsheet;

import com.google.common.base.Throwables;
import gov.va.vamf.service.clio.application.representations.UniqueTermId;
import gov.va.vamf.service.clio.flowsheet.cache.*;
import gov.va.vamf.service.clio.flowsheet.representations.*;
import gov.va.vamf.service.clio.infrastructure.security.UserServices;
import gov.va.vamf.service.clio.infrastructure.vista.*;
import org.slf4j.*;

/**
 * Loader used to load a list of Flowsheets that are available for a facility. Load method is thread safe.
 */
public class FlowsheetItemsLoader implements Loader<CacheKey<String>, FlowsheetList> {
    private static Logger logger = LoggerFactory.getLogger(FlowsheetItemsLoader.class);

    @Override
    public FlowsheetList load(CacheKey<String> key) {
        logger.debug("Loading list of available flowsheets for facility {}.", key.value);

        FlowsheetList items = new FlowsheetList();

        ClioResultRecords resultRecords = getClioResultRecords(key.vistaService);

        while (resultRecords.hasNext()) {
            ClioResultRecord resultRecord = resultRecords.next();

            if (flowsheetActive(resultRecord))
                items.flowsheetItems.add(createFlowsheetItem(resultRecord));
        }

        return items;
    }

    public ClioResultRecords getClioResultRecords(VistaService vistaService) {
        try {
            return vistaService.makeClioCall("GetFlowsheetList");
        } catch (Exception e) {
            logger.error("Unable to load flowsheet list.", e);

            throw new FlowsheetException(500, "Cannot return flowsheet list. Unable to retrieve list for your facility.",
                    Throwables.getRootCause(e).getMessage());
        }
    }

    private boolean flowsheetActive(ClioResultRecord resultRecord) {
        return resultRecord.getString("ACTIVE").equalsIgnoreCase("1");
    }

    private FlowsheetItem createFlowsheetItem(ClioResultRecord resultRecord) {
        UniqueTermId termId = new UniqueTermId(resultRecord.getString("FLOWSHEET_ID"), resultRecord.getString("DISPLAY_NAME"));
        FlowsheetItem item = new FlowsheetItem(termId);

        item.comment = resultRecord.getString("COMMENT");

        return item;
    }
}
